# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import re
from collections import defaultdict
from datetime import datetime

import six
from iso8601 import parse_date


MAX_DIFF_FROM_EVENT = 15


def get_thread_suburban_key(number, first_station):
    return '{}__{}'.format(number, first_station.id)


def get_suburban_key_number_station(suburban_key):
    return suburban_key.split('__')


def get_rtstation_key(rts):
    return '{}'.format(rts.station.id)


def get_thread_type_and_clock_dir(thread):
    if 'MCZK' in thread.uid:
        if 'по часовой стрелке' in thread.title:
            clock_direction = ClockDirection.CLOCK_WISE
        elif 'против часовой стрелки' in thread.title:
            clock_direction = ClockDirection.C_CLOCK_WISE
        return ThreadEventsTypeCodes.MCZK, clock_direction
    return ThreadEventsTypeCodes.SUBURBAN, None


def prefix_for_dict_keys(dct, prefix):
    """
    Используется для формирования ключей-запросов к монге, когда ключ - вложенный словарь.

    >>> prefix_for_dict_keys({'a': 1, 'b': 2}, 'key.')
    {'key.a': 1, 'key.b': 2}

    """
    for key in dct.keys():
        dct[prefix + key] = dct.pop(key)

    return dct


@six.python_2_unicode_compatible
class ThreadKey(object):
    def __init__(self, thread_key, thread_start_date, thread_type=None, clock_direction=None):
        assert isinstance(thread_start_date, datetime)

        self.thread_key = thread_key
        self.thread_start_date = thread_start_date
        self.thread_type = thread_type
        self.clock_direction = clock_direction

    def __eq__(self, other):
        return self.compare_str == other.compare_str

    @property
    def compare_str(self):
        if not hasattr(self, '_compare_str'):
            self._compare_str = '{thread_key}___{thread_start_date}___{thread_type}___{clock_direction}'.format(
                thread_key=self.thread_key,
                thread_start_date=self.thread_start_date.isoformat(),
                thread_type=self.thread_type,
                clock_direction=self.clock_direction
            )

        return self._compare_str

    @property
    def get_hash(self):
        # объект не должен меняться
        if not hasattr(self, '_hash'):
            self._hash = hash(self.compare_str)

        return self._hash

    def __hash__(self):
        return self.get_hash

    def __str__(self):
        return self.compare_str

    def to_str(self):
        return self.compare_str

    @classmethod
    def from_str(cls, str_key):
        match = re.match(
            '^(?P<thread_key>.+)___(?P<start_date>.+)___(?P<thread_type>.+)___(?P<clock_direction>.+)$',
            str_key
        )
        if match:
            iso_dt = parse_date(match.group('start_date'), default_timezone=None)
            thread_type = match.group('thread_type') if match.group('thread_type') != 'None' else None
            clock_direction = int(match.group('clock_direction')) if match.group('clock_direction') != 'None' else None

            return cls(
                thread_key=match.group('thread_key'),
                thread_start_date=iso_dt,
                thread_type=thread_type,
                clock_direction=clock_direction
            )
        else:
            raise ValueError('Invalid str_key: {}'.format(str_key))

    def to_mongo_dict(self):
        """ Словарь для поиска объекта по models.ThreadKey """
        mongo_dict = {
            'thread_key': self.thread_key,
            'thread_start_date': self.thread_start_date,
        }

        if self.clock_direction:
            mongo_dict['thread_type'] = self.thread_type
            mongo_dict['clock_direction'] = self.clock_direction

        return mongo_dict


@six.python_2_unicode_compatible
class ThreadStationKey(ThreadKey):
    def __init__(self, thread_key, thread_start_date, station_key, thread_type=None, clock_direction=None,
                 arrival=None, departure=None):
        self.station_key = station_key
        self.arrival = arrival
        self.departure = departure
        super(ThreadStationKey, self).__init__(thread_key, thread_start_date, thread_type, clock_direction)

    @property
    def compare_str(self):
        if not hasattr(self, '_compare_str'):
            self._compare_str = ('{thread_key}___{thread_start_date}___{station_key}___{arrival}___{departure}'
                                 '___{thread_type}___{clock_direction}').format(
                thread_key=self.thread_key,
                thread_start_date=self.thread_start_date.isoformat(),
                station_key=self.station_key,
                arrival=self.arrival,
                departure=self.departure,
                thread_type=self.thread_type,
                clock_direction=self.clock_direction
            )

        return self._compare_str

    def __str__(self):
        if self.arrival is None and self.departure is None:
            raise ValueError(u"Can't convert key to str: empty times")
        format_str = ('{thread_key}___{thread_start_date}___{station_key}___{arrival}___{departure}'
                      '___{thread_type}___{clock_direction}')
        return format_str.format(
            thread_key=self.thread_key,
            thread_start_date=self.thread_start_date.isoformat(),
            station_key=self.station_key,
            arrival=self.arrival,
            departure=self.departure,
            thread_type=self.thread_type,
            clock_direction=self.clock_direction
        )

    def to_str(self):
        if self.arrival is None and self.departure is None:
            raise ValueError(u"Can't convert key to str: empty times")
        format_str = ('{thread_key}___{thread_start_date}___{station_key}___{arrival}___{departure}'
                      '___{thread_type}___{clock_direction}')
        return format_str.format(
            thread_key=self.thread_key,
            thread_start_date=self.thread_start_date.isoformat(),
            station_key=self.station_key,
            arrival=self.arrival,
            departure=self.departure,
            thread_type=self.thread_type,
            clock_direction=self.clock_direction
        )

    @classmethod
    def from_str(cls, str_key):
        match = re.match(
            '^(?P<thread_key>.+)___(?P<start_date>.+)___(?P<station_key>.+)___(?P<arrival>.+)___(?P<departure>.+)'
            '___(?P<thread_type>.+)___(?P<clock_direction>.+)$',
            str_key
        )

        if match:
            iso_dt = parse_date(match.group('start_date'), default_timezone=None)
            arrival = int(match.group('arrival')) if match.group('arrival') != 'None' else None
            departure = int(match.group('departure')) if match.group('departure') != 'None' else None
            thread_type = match.group('thread_type') if match.group('thread_type') != 'None' else None
            clock_direction = int(match.group('clock_direction')) if match.group('clock_direction') != 'None' else None

            return cls(
                thread_key=match.group('thread_key'),
                thread_start_date=iso_dt,
                station_key=match.group('station_key'),
                arrival=arrival,
                departure=departure,
                thread_type=thread_type,
                clock_direction=clock_direction
            )
        else:
            raise ValueError('Invalid str_key: {}'.format(str_key))

    def to_mongo_dict(self):
        dct = super(ThreadStationKey, self).to_mongo_dict()
        dct['station_key'] = self.station_key
        dct['arrival'] = self.arrival
        dct['departure'] = self.departure
        return dct

    def get_thread_key(self):
        return ThreadKey(self.thread_key, self.thread_start_date, self.thread_type, self.clock_direction)


def get_threads_suburban_keys(threads):
    from common.apps.suburban_events.models import SuburbanKey

    suburban_keys = {}
    for sk in SuburbanKey.objects.filter(thread__in=set(threads)):
        suburban_keys[sk.thread_id] = sk.key

    return suburban_keys


def get_threads_by_suburban_keys(suburban_keys):
    from common.apps.suburban_events.models import SuburbanKey

    by_key = defaultdict(list)
    for sk in SuburbanKey.objects.filter(key__in=suburban_keys).select_related('thread'):
        by_key[sk.key].append(sk.thread)

    return by_key


def find_state_for_rts(station_states, rts_arrival, rts_departure):
    """
    Если одному ключу соотвествуют несколько station_states (т.е. нитка проходит через
    одну станцию 2+ раза), то нужно выбрать наиболее подходящее событие.
    Выбираем подходящее по близости по времени.
    """

    if not station_states:
        return None

    station_state = None
    state_diffs = []
    for sts in station_states:
        if rts_departure is not None:  # поиск подходящего события для всех станций, кроме последней
            # точно последняя станция, не подходит
            if sts.get('departure') is None:
                continue

            # точно первая станция
            if rts_arrival is None and sts.get('arrival') is None:
                state_diffs.append((0, sts))
                break

            # считаем разницу во времени между событиями и нашим rts, чтобы решить, какое событие наше
            diff = abs(sts.get('departure') - rts_departure)
            state_diffs.append((diff, sts))
        else:  # для последней станции
            # точно последняя станция, подходит
            if sts.get('departure') is None:
                state_diffs.append((0, sts))
                break

            # считаем разницу во времени между событиями и нашим rts, чтобы решить, какое событие наше
            diff = abs(sts.get('arrival') - rts_arrival)
            state_diffs.append((diff, sts))

    if state_diffs:
        diff, station_state = sorted(state_diffs)[0]
        if station_state.get('passed_several_times') and diff > MAX_DIFF_FROM_EVENT:
            # Т.к. в нашей нитке может быть дубль этой станции, и событие слишком далеко,
            # считаем, что это событие не относится к нашему дублю станции.
            station_state = None

    return station_state


class EventStateType(object):
    FACT = 'fact'
    FACT_INTERPOLATED = 'fact_interpolated'
    POSSIBLE_DELAY = 'possible_delay'
    POSSIBLE_OK = 'possible_ok'  # ожидается по расписанию
    UNDEFINED = 'undefined'
    CANCELLED = 'cancelled'


class ThreadEventsTypeCodes(object):
    SUBURBAN = None
    MCZK = 'mczk'


class EventStateSubType(object):
    FACT = 'fact'
    NO_NEXT_DATA = 'no_next_data'
    NO_DATA = 'no_data'
    NO_DATA_FIRST_STATION = 'no_data_first_station'
    TRAIN_TURNOVER = 'train_turnover'
    MOVISTA_CANCEL = 'movista_cancel'


class ClockDirection(object):
    CLOCK_WISE = 1
    C_CLOCK_WISE = 0


class LightMongoEngineDoc(object):
    def __init__(self, data):
        self.data = data

    def __getattr__(self, item):
        value = self.data[item]
        if isinstance(value, (dict, list)):
            return LightMongoEngineDoc(value)
        else:
            return self.data[item]

    def __len__(self):
        return len(self.data)

    def __hash__(self):
        return hash(self.data)

    def __eq__(self, other):
        return self.data == other

    def __ne__(self, other):
        return self.data != other

    def __iter__(self):
        return ((LightMongoEngineDoc(o) if isinstance(o, (dict, list)) else o) for o in iter(self.data))


def light_mongoengine_query(query):
    collection = query._document._get_collection()
    return [LightMongoEngineDoc(d) for d in collection.find(query._query)]


def collect_rts_by_threads(rtstations, collect_stations=False):
    thread_paths = defaultdict(list)
    for rts in rtstations:
        # rts отсортированы по id, поэтому станции окажутся в списке по ходу следования
        thread_paths[rts.thread].append(rts.station if collect_stations else rts)
    return thread_paths


def collect_threads_by_number(thread_paths):
    threads_by_number = defaultdict(set)
    for thread in thread_paths:
        for number_part in thread.number.split('/'):
            if not number_part.strip():
                continue

            threads_by_number[number_part].add(thread)

    return threads_by_number
