# coding: utf-8

from __future__ import unicode_literals

import logging
import re

import six
from django.core.exceptions import ValidationError
from django.core.validators import RegexValidator
from django.db import models
from django.utils.encoding import force_text
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _

from common.apps.train_order.enums import CoachType
from common.db.mds.s3_storage import mds_s3_media_storage
from common.models_utils.i18n import L_field
from common.precache.manager import PrecachingManager
from common.utils.fields import CodeCharField, TrimmedCharField
from common.utils.text import split_string


log = logging.getLogger(__name__)

COACH_TYPE_CHOICES = (
    # категории вагонов
    (CoachType.SUITE.value, _('СВ')),
    (CoachType.COMPARTMENT.value, _('купе')),
    (CoachType.PLATZKARTE.value, _('плацкарт')),
    (CoachType.SITTING.value, _('сидячие')),
    (CoachType.COMMON.value, _('общий')),
    (CoachType.SOFT.value, _('мягкий'))
)


@six.python_2_unicode_compatible
class PlacePriceRules(models.Model):
    rule_name = TrimmedCharField(_('Название правила'), max_length=1024, blank=True, default='')
    coach_type = models.CharField(_('Тип вагона'), max_length=20, choices=COACH_TYPE_CHOICES,
                                  blank=True, null=False, default=CoachType.PLATZKARTE.value)
    place_name = TrimmedCharField(_('Название мест'), max_length=1024, help_text='например, верхние боковушки',
                                  blank=True, null=False, default='')
    place_numbers = TrimmedCharField(_('Номера мест'), max_length=1024, blank=True, null=False, default='')

    priority = models.IntegerField(verbose_name=_('Приоритет'),
                                   help_text=_('Приоритет правила (чем больше тем важнее)'), null=False, default=0)
    price_percent = models.FloatField(verbose_name=_('Процент'),
                                      help_text=_('Процент полной цены билета'), null=False, default=100)

    train_number = TrimmedCharField(_('Номер поезда'), help_text=_('Регулярное выражение'),
                                    max_length=1024, blank=True, null=False, default='')

    station_from = models.ForeignKey('www.Station', verbose_name=_('с какой станции'),
                                     null=True, blank=True, related_name='station_from_set')
    station_to = models.ForeignKey('www.Station', verbose_name=_('до какой станции'),
                                   null=True, blank=True, related_name='station_to_set')

    service_class = TrimmedCharField(_('Класс обслуживания ЖД'), help_text=_('класс обслуживания ЖД - 2Э, ...'),
                                     max_length=1024, blank=True, null=False, default='')
    owner = TrimmedCharField(_('Владельцы вагонов (через запятую)'),
                             help_text=_('Например, ФПК, ДОСС, БЧ, ТВЕРСК'),
                             max_length=1024, blank=True, null=False, default='')

    departure_period_begin = models.DateField(verbose_name=_('Период отправления поезда (начало)'),
                                              null=True, blank=True)
    departure_period_end = models.DateField(verbose_name=_('Период отправления поезда (конец)'),
                                            null=True, blank=True)
    sale_period_begin = models.IntegerField(verbose_name=_('Начало продаж по акции (дней до отправления)'),
                                            null=True, blank=True)
    sale_period_end = models.IntegerField(verbose_name=_('Конец продаж по акции (дней до отправления'),
                                          null=True, blank=True)

    is_deluxe = models.NullBooleanField(verbose_name=_('Фирменный поезд'), null=True)
    _coach_numbers = TrimmedCharField(verbose_name=_('Номера вагонов (через запятую)'),
                                      max_length=1024, null=False, blank=True, default='',
                                      db_column='coach_numbers')
    is_dinamic_tariff = models.NullBooleanField(verbose_name=_('Динамический тариф'), null=True)
    can_apply_for_child = models.BooleanField(verbose_name=_('Действует для детского билета'),
                                              null=False, default=True, blank=False)

    @cached_property
    def place_numbers_set(self):
        groups = split_string(self.place_numbers)
        place_numbers = set()
        for group in groups:
            values = split_string(group, '-')
            if len(values) == 1:
                place_numbers.add(int(values[0]))
            elif len(values) == 2:
                start = int(values[0])
                end = int(values[1])
                if start < end:
                    place_numbers.update(range(start, end + 1))
                else:
                    log.error('Некорректный набор мест для PlacePriceRules, id = {}'.format(self.id))

        return place_numbers

    @cached_property
    def owners(self):
        return [owner.upper() for owner in split_string(self.owner)]

    @cached_property
    def coach_numbers(self):
        return [int(number) for number in split_string(self._coach_numbers)]

    def check_owner(self, owner):
        return owner.upper() in self.owners

    def check_train_number(self, train_number):
        if not self.train_number:
            return False

        train_number_re = re.compile(self.train_number, flags=re.U | re.I)
        return bool(train_number_re.match(train_number))

    def check_coach_number(self, coach_number):
        return coach_number in self.coach_numbers

    def __str__(self):
        return '{}'.format(self.rule_name)

    def clean(self):
        self._validate_rule_intersections()

    def _validate_rule_intersections(self):
        exclude_fields = ['id', 'rule_name', 'place_name', 'place_numbers', 'priority', 'price_percent',
                          'owner', '_coach_numbers']
        params = {
            f.name: getattr(self, f.name)
            for f in PlacePriceRules._meta.get_fields()
            if f.name not in exclude_fields and f.concrete
        }
        queryset = PlacePriceRules.objects.filter(**params)
        if self.id:
            queryset = queryset.exclude(id=self.id)

        intersections = []
        for rule in queryset:
            if set(self.owners) != set(rule.owners):
                continue
            if set(self.coach_numbers) != set(rule.coach_numbers):
                continue

            place_intersection = sorted(rule.place_numbers_set & self.place_numbers_set)
            if place_intersection:
                intersections.append((rule, place_intersection))

        if intersections:
            intersection_text = '; '.join('{}-{} places: {}'.format(rule.id, rule.rule_name,
                                                                    ', '.join(map(force_text, places)))
                                          for rule, places in intersections)
            raise ValidationError({
                'place_numbers': ValidationError('Обнаружено пересечение с правилами: %s',
                                                 params=(intersection_text,))
            })

    class Meta:
        verbose_name = _('правило расчета тарифов на места в вагоне')
        verbose_name_plural = _('правила расчета тарифов на места в вагоне')
        app_label = 'train'


@six.python_2_unicode_compatible
class CoachFacility(models.Model):
    name = models.CharField(_('Название для админки'), max_length=1024, blank=True, default='')
    code = CodeCharField(_('Код'), max_length=100, validators=[
        RegexValidator(r'[A-Za-z][_A-Za-z0-9]*', 'Код должен содержать только латинские буквы и цифры и знак "_"')
    ], unique=True)
    L_title = L_field(_('Название для отображения'), base_lang='ru', base_field_critical=True)

    icon = models.FileField(verbose_name=_('иконка в svg'),
                            storage=mds_s3_media_storage,
                            upload_to='data/train/coach-facility',
                            null=False, blank=False)

    order = models.IntegerField(_('Порядок отображения'), blank=False, null=False, default=0)

    TANKER_L_FIELDS = ['title']
    TANKER_KEY_FIELD = 'code'

    def __str__(self):
        return '{} {}'.format(self.code, self.name)

    class Meta:
        verbose_name = _('Удобство в вагоне')
        verbose_name_plural = _('Удобства в вагоне')


class ServiceClass(models.Model):
    name = models.CharField(_('Название для админки'), max_length=1024, blank=True, default='')
    code = CodeCharField(_('Коды классов обслуживания (через запятую), например "2Э, 2С"'), max_length=100)
    L_title = L_field(_('Название для отображения (короткое)'), base_lang='ru', base_field_critical=True,
                      default='', null=False, blank=True)
    L_description = L_field(_('Описание класса обслуживания для отображения'), field_cls=models.TextField,
                            base_lang='ru', base_field_critical=False, default='', null=False, blank=True)
    coach_category = models.CharField(_('Категория вагона'), max_length=20, choices=COACH_TYPE_CHOICES,
                                      blank=True, null=False, default='')
    coach_owner = models.CharField(_('Владельцы вагонов (через запятую), например, ФПК, РЖД, ДОСС'), max_length=100,
                                   blank=True, null=False, default='')
    brand_title = models.CharField(_('Название бренда, например, САПСАН, ЛАСТОЧКА, SWIFT,..'), max_length=100,
                                   blank=True, null=False, default='')
    is_firm_coach = models.NullBooleanField(_('Вагон является фирменным'), null=True, blank=True, default=None)
    train_number = models.CharField(_('Номер поезда'), help_text=_('Регулярное выражение'),
                                    max_length=1024, blank=True, null=False, default='')
    two_storey = models.NullBooleanField(_('Двухэтажный вагон'), null=True, blank=True, default=None)
    start_date = models.DateField(_('Дата начала действия класса обслуживания (может быть пустой)'),
                                  blank=True, null=True, default=None)
    end_date = models.DateField(_('Дата окончания действия класса обслуживания (может быть пустой)'),
                                blank=True, null=True, default=None)
    facilities = models.ManyToManyField(CoachFacility, verbose_name=_('Услуги'), blank=True)

    TANKER_L_FIELDS = ['title', 'description']

    @cached_property
    def coach_owners(self):
        return [owner.upper() for owner in split_string(self.coach_owner)]

    @cached_property
    def codes(self):
        return [code for code in split_string(self.code)]

    class Meta:
        verbose_name = _('Класс обслуживания')
        verbose_name_plural = _('Классы обслуживания')
        app_label = 'train'


@six.python_2_unicode_compatible
class UFSErrorDescription(models.Model):
    name = models.CharField(_('Название для админки'), max_length=1024, blank=True, null=False, default='')
    ufs_code = CodeCharField(_('Код ошибки УФС'), max_length=20, null=False, blank=False, unique=True)
    ufs_description = models.TextField(_('Оригинальное описание ошибки от УФС'), null=False, blank=True, default='')
    L_description = L_field(_('Описание ошибки для отображения'), field_cls=models.TextField,
                            base_lang='ru', base_field_critical=False, default='', null=False, blank=True)
    show = models.BooleanField(_('Показывать вместо стандартного сообщения об ошибке'), null=False, default=False)

    TANKER_L_FIELDS = ['description']

    def __str__(self):
        return '{} {} {}'.format(self.name, self.ufs_code, self.ufs_description)

    class Meta:
        verbose_name = _('Описание ошибки УФС')
        verbose_name_plural = _('Описания ошибок УФС')


@six.python_2_unicode_compatible
class UFSNewOrderBlackList(models.Model):
    number = TrimmedCharField(_('Номера'), max_length=500, null=False, blank=False,
                              help_text=_('Список номеров через запятую'))
    comment = models.TextField(_('Комментарий'), null=True, blank=True)

    def __str__(self):
        return 'Исключение поездов с номерами: {}'.format(self.number)

    @cached_property
    def numbers(self):
        return {s.strip() for s in self.number.split(',') if s.strip()}

    class Meta:
        verbose_name = _('Исключение поездов из новой продажи')
        verbose_name_plural = _('Исключения поездов из новой продажи')


class TariffInfo(models.Model):
    FULL_CODE = 'full'
    BABY_CODE = 'baby'
    CHILD_CODE = 'child'

    code = CodeCharField(_('Код'), max_length=100, validators=[
        RegexValidator(r'[A-Za-z][_A-Za-z0-9]*', 'Код должен содержать только латинские буквы и цифры и знак "_"')
    ], unique=True)
    L_title = L_field(_('Название для отображения'), base_lang='ru', base_field_critical=True)

    need_document = models.BooleanField(_('Необходим документ'), blank=False, null=False, default=False)
    without_place = models.BooleanField(_('Тариф без места'), blank=False, null=False, default=False)

    min_age = models.PositiveIntegerField(_('Нижняя граница возраста пассажира'), blank=False, null=False, default=0)
    min_age_includes_birthday = models.BooleanField(_('Нижняя граница включает день рождения'),
                                                    blank=False, null=False, default=False)
    max_age = models.PositiveIntegerField(_('Верхняя граница возраста пассажира'), blank=False, null=False, default=150)
    max_age_includes_birthday = models.BooleanField(_('Верхняя граница включает день рождения'),
                                                    blank=False, null=False, default=False)

    ufs_request_code = TrimmedCharField(_('Код тарифа для запроса в УФС'), max_length=500, null=False)
    ufs_response_codes = TrimmedCharField(_('Коды тарифа из ответа от УФС'), help_text=_('Список через запятую'),
                                          max_length=500, null=False)

    im_request_code = TrimmedCharField(_('Код тарифа для запроса в ИМ'), max_length=500, null=False)
    im_response_codes = TrimmedCharField(_('Коды тарифа из ответа от ИМ'), help_text=_('Список через запятую'),
                                         max_length=500, null=False)

    @property
    def ufs_response_codes_list(self):
        return list(map(six.text_type.strip, self.ufs_response_codes.split(','))) if self.ufs_response_codes else []

    @property
    def im_response_codes_list(self):
        return list(map(six.text_type.strip, self.im_response_codes.split(','))) if self.im_response_codes else []

    class Meta:
        verbose_name = _('Информация о тарифе')
        verbose_name_plural = _('Информация о тарифах')


class Facility(models.Model):
    name = models.CharField(_('Название для админки'), max_length=1024, blank=True, default='')
    code = CodeCharField(_('Код'), max_length=100, validators=[
        RegexValidator(r'[A-Za-z][_A-Za-z0-9]*', 'Код должен содержать только латинские буквы и цифры и знак "_"')
    ], unique=True)
    im_code = CodeCharField(_('Код в системе IM'), max_length=100, unique=True)

    L_title = L_field(_('Название для отображения'), base_lang='ru', base_field_critical=True)
    for_place = models.BooleanField(_('Удобство связанное с местом'), max_length=1024, blank=True, null=False,
                                    default=False)
    order = models.IntegerField(_('Порядок отображения'), blank=False, null=False, default=0)

    TANKER_L_FIELDS = ['title']
    TANKER_KEY_FIELD = 'code'

    # Константы, чтобы не хардкодить в коде числа
    EATING_ID = 1
    PAPER_ID = 2
    TV_ID = 3
    TRANSFER_ID = 4
    CONDITIONER_ID = 5
    BED_CLOTHES_ID = 6
    SANITARY_TOOLBOX_ID = 7
    WIFI_ID = 8
    NEAR_TOILET_ID = 9
    SIDE_PLACE_ID = 10
    UPPER_PLACE_ID = 11

    objects = PrecachingManager(keys=['pk', 'code', 'im_code'])

    class Meta:
        verbose_name = _('Удобство')
        verbose_name_plural = _('Удобства')
        app_label = 'train'


class CoachType(models.Model):
    name = models.CharField(_('Название для админки'), max_length=1024, blank=True, default='')
    code = CodeCharField(_('Код'), max_length=100, validators=[
        RegexValidator(r'[A-Za-z][_A-Za-z0-9]*', 'Код должен содержать только латинские буквы и цифры и знак "_"')
    ], unique=True)
    im_code = CodeCharField(_('Код в системе IM'), max_length=100, unique=True)
    L_title = L_field(_('Название для отображения'), base_lang='ru', base_field_critical=True)
    order = models.IntegerField(_('Порядок отображения'), blank=False, null=False, default=0)

    TANKER_L_FIELDS = ['title']
    TANKER_KEY_FIELD = 'code'

    # Константы, чтобы не хардкодить в коде числа
    COMMON_ID = 1
    SITTING_ID = 2
    PLATZKARTE_ID = 3
    COMPARTMENT_ID = 4
    SOFT_ID = 5
    SUITE_ID = 6

    objects = PrecachingManager(keys=['pk', 'code', 'im_code'])

    class Meta:
        verbose_name = _('Тип вагона')
        verbose_name_plural = _('Типы вагонов')
        app_label = 'train'
