# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import socket
from functools import wraps

from celery import shared_task
from django.conf import settings
from mongolock import MongoLock

from common.db.mongo import ConnectionProxy
from common.utils.lock import LOCK_COLLECTION_NAME

log = logging.getLogger(__name__)

database = ConnectionProxy(settings.LOCK_DATABASE_NAME)


def single_launch_task(expire=60, *args, **kwargs):
    """
    В случае, когда у нас запущен кластер celery с celery-beat (или cron'ом), запущенным на каждой машине -
    мы оказываемся в ситуации, когда каждая периодическая задача будет запланирована N раз (где N размер кластера).

    Чтобы быть уверенным, что задача была выполнена только один раз (дедупликация), используем MongoLock

    :param expire: период в секундах на который блокируем запуск задач
    :return:
    """
    def wrapper(f):
        @shared_task(*args, **kwargs)
        @wraps(f)
        def actual_wrapper(*args, **kwargs):
            key = _build_unique_launch_key(f)
            lock = MongoLock(collection=getattr(database, LOCK_COLLECTION_NAME))
            if lock.lock(key, socket.gethostname(), expire=expire):
                return f(*args, **kwargs)
            else:
                log.info('Task with key %s already started', key)
                return None
        return actual_wrapper
    return wrapper


def _build_unique_launch_key(f):
    return '{}.{}'.format(f.__module__, f.__name__)
