# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import zlib
import base64
import json
from collections import OrderedDict

from lxml import etree
from django.utils.encoding import smart_str

from common.utils.caching import cache_method_result


class BaseBlock(object):
    def to_xml_file(self, filepath):
        tree = self.get_element()

        with open(filepath, 'w') as f:
            f.write(etree.tostring(tree, xml_declaration=True, encoding='utf-8', pretty_print=True))

    def to_unicode_xml(self):
        tree = self.get_element()

        return etree.tostring(tree, encoding=unicode, pretty_print=True)

    def get_element(self):
        raise NotImplementedError()


class ChannelBlock(BaseBlock):
    def __init__(self, t_type, version='1.0', station_code_system='local', carrier_code_system='local',
                 vehicle_code_system='local', timezone='start_station',
                 country_code_system='IATA', region_code_system='yandex', settlement_code_system='yandex',
                 t_subtype=None):

        self._groups = []

        self.t_type = t_type
        self.t_subtype = t_subtype
        self.station_code_system = station_code_system
        self.carrier_code_system = carrier_code_system
        self.vehicle_code_system = vehicle_code_system
        self.timezone = timezone
        self.version = version

        self.country_code_system = country_code_system
        self.region_code_system = region_code_system
        self.settlement_code_system = settlement_code_system

    def get_element(self):
        channel_el = etree.Element('channel')
        channel_el.set('version', self.version)
        channel_el.set('t_type', self.t_type)
        if self.t_subtype:
            channel_el.set('subtype', self.t_subtype)
        channel_el.set('station_code_system', self.station_code_system)
        channel_el.set('carrier_code_system', self.carrier_code_system)
        channel_el.set('vehicle_code_system', self.vehicle_code_system)
        channel_el.set('timezone', self.timezone)

        channel_el.set('country_code_system', self.country_code_system)
        channel_el.set('region_code_system', self.region_code_system)
        channel_el.set('settlement_code_system', self.settlement_code_system)

        for group in self._groups:
            group_el = group.get_element()
            channel_el.append(group_el)

        return channel_el

    def add_group_block(self, group):
        self._groups.append(group)


class GroupBlock(BaseBlock):
    optional_attributes = ('t_type', 'station_code_system', 'carrier_code_system', 'vehicle_code_system', 'timezone')

    def __init__(self, channel, title, code, t_type=None, station_code_system=None, carrier_code_system=None,
                 vehicle_code_system=None, timezone=None):

        self.channel = channel

        self._stations = OrderedDict()
        self._carriers = OrderedDict()
        self._vehicles = OrderedDict()
        self._fares = OrderedDict()
        self._threads = []

        self.title = title
        self.code = code

        self.timezone = timezone or channel.timezone
        self.t_type = t_type or channel.t_type
        self.station_code_system = station_code_system or channel.station_code_system
        self.carrier_code_system = carrier_code_system or channel.carrier_code_system
        self.vehicle_code_system = vehicle_code_system or channel.vehicle_code_system

    def get_element(self):
        group_el = etree.Element('group')
        group_el.set('title', self.title)
        group_el.set('code', self.code)

        for attr in self.optional_attributes:
            if getattr(self, attr) != getattr(self.channel, attr, None):
                group_el.set(attr, getattr(self, attr))

        if self._stations:
            stations_el = etree.SubElement(group_el, 'stations')
            for station in self._stations.values():
                stations_el.append(station.get_element())

        if self._vehicles:
            vehicles_el = etree.SubElement(group_el, 'vehicles')
            for vehicle in self._vehicles.values():
                vehicles_el.append(vehicle.get_element())

        if self._carriers:
            carriers_el = etree.SubElement(group_el, 'carriers')
            for thread in self._carriers.values():
                carriers_el.append(thread.get_element())

        threads_el = etree.SubElement(group_el, 'threads')
        if self._threads:
            for thread in self._threads:
                threads_el.append(thread.get_element())

        if self._fares:
            fares_el = etree.SubElement(group_el, 'fares')
            for fare in self._fares.values():
                fares_el.append(fare.get_element())

        return group_el

    def add_station_block(self, station):
        key = station.key
        if key not in self._stations:
            self._stations[key] = station

        return self._stations[key]

    @cache_method_result
    def add_local_station(self, title):
        code = unicode(len(self._stations) + 1)

        station = StationBlock(group=self, title=title, code=code, code_system='local')

        return self.add_station_block(station)

    def add_station(self, title, code, code_system=None):
        station = StationBlock(group=self, title=title, code=code, code_system=code_system)

        return self.add_station_block(station)

    def add_carrier_block(self, carrier):
        key = carrier.key
        if key not in self._carriers:
            self._carriers[key] = carrier

        return self._carriers[key]

    @cache_method_result
    def add_local_carrier(self, title):
        code = unicode(len(self._carriers) + 1)

        carrier = CarrierBlock(group=self, title=title, code=code, code_system='local')

        return self.add_carrier_block(carrier)

    def add_carrier(self, title, code, code_system=None, phone=None, address=None, country_code=None):
        carrier = CarrierBlock(group=self, title=title, code=code, code_system=code_system,
                               phone=phone, address=address, country_code=country_code)

        return self.add_carrier_block(carrier)

    def add_vehicle_block(self, vehicle):
        key = vehicle.key
        if key not in self._vehicles:
            self._vehicles[key] = vehicle

        return self._vehicles[key]

    @cache_method_result
    def add_local_vehicle(self, title, recommended_title=None):
        code = unicode(len(self._vehicles) + 1)

        vehicle = VehicleBlock(group=self, title=title, code=code,
                               code_system='local', recommended_title=recommended_title)

        return self.add_vehicle_block(vehicle)

    def add_vehicle(self, title, code, code_system=None, recommended_title=None):
        vehicle = VehicleBlock(group=self, title=title, code=code,
                               code_system=code_system, recommended_title=recommended_title)

        return self.add_vehicle_block(vehicle)

    def add_local_fare(self):
        code = u'#auto_generated_code#_' + unicode(len(self._fares) + 1)
        return self.add_fare(code)

    def add_fare(self, code):
        fare = self._fares.get(code, None)
        if fare is None:
            fare = FareBlock(group=self, code=code)
            self._fares[code] = fare
        return self._fares[code]

    def add_thread_block(self, thread):
        self._threads.append(thread)

        return thread


class StationBlock(BaseBlock):
    def __init__(self, group, title, code, code_system=None, recommended_title=None):
        self.group = group
        self.title = title
        self.code = code
        self.lat = None
        self.lon = None
        self.recommended_title = recommended_title
        self.geocode_title = None
        self.code_system = code_system or group.station_code_system
        self.data = None

        self.country_code = None
        self.region_code = None
        self.settlement_code = None

        self._legacy_stations = set()

    def get_element(self):
        station_el = etree.Element('station')
        station_el.set('title', self.title)
        station_el.set('code', self.code)

        if self.lat:
            station_el.set('lat', self.lat)

        if self.lon:
            station_el.set('lon', self.lon)

        if self.country_code:
            station_el.set('country_code', self.country_code)

        if self.region_code:
            station_el.set('region_code', self.region_code)

        if self.settlement_code:
            station_el.set('settlement_code', self.settlement_code)

        if self.recommended_title:
            station_el.set('recommended_title', self.recommended_title)

        if self.geocode_title:
            station_el.set('_geocode_title', self.geocode_title)

        if self.code_system != self.group.station_code_system:
            station_el.set('code_system', self.code_system)

        if self._legacy_stations:
            for title, code in self._legacy_stations:
                etree.SubElement(station_el, 'legacy_station', {
                    'type': 'raw',
                    'code': code,
                    'title': title
                })

        if self.data:
            data_el = etree.SubElement(station_el, 'data')
            data_el.text = self.data

        return station_el

    def set_json_data(self, data):
        self.data = json.dumps(data)

    def add_legacy_station(self, title, code):
        self._legacy_stations.add((title, code))

    @property
    def key(self):
        return u"{}#*#{}#*#{}".format(self.title, self.code, self.code_system)


class CarrierBlock(BaseBlock):
    optional_attributes = ('phone', 'address', 'country_code')

    def __init__(self, group, title, code, code_system=None, phone=None, address=None, country_code=None):
        self.group = group
        self.title = title
        self.code = code
        self.code_system = code_system or group.carrier_code_system
        self.phone = phone
        self.address = address
        self.country_code = country_code

    def get_element(self):
        carrier_el = etree.Element('carrier')
        carrier_el.set('title', self.title)
        carrier_el.set('code', self.code)

        if self.code_system != self.group.carrier_code_system:
            carrier_el.set('code_system', self.code_system)

        for attr in self.optional_attributes:
            if getattr(self, attr) is not None:
                carrier_el.set(attr, getattr(self, attr))

        return carrier_el

    @property
    def key(self):
        return u"{}#*#{}#*#{}".format(self.title, self.code, self.code_system)


class VehicleBlock(BaseBlock):
    def __init__(self, group, title, code, code_system=None, recommended_title=None):
        self.group = group
        self.title = title
        self.recommended_title = recommended_title
        self.code = code
        self.code_system = code_system or group.vehicle_code_system

    def get_element(self):
        vehicle_el = etree.Element('vehicle')
        vehicle_el.set('title', self.title)
        vehicle_el.set('code', self.code)

        if self.code_system != self.group.vehicle_code_system:
            vehicle_el.set('code_system', self.code_system)

        if self.recommended_title:
            vehicle_el.set('recommended_title', self.recommended_title)

        return vehicle_el

    @property
    def key(self):
        return u"{}#*#{}#*#{}".format(self.title, self.code, self.code_system)


class ThreadBlock(BaseBlock):
    optional_attributes = ('title', 'number')
    optional_inheritable_attributes = ('t_type', 'station_code_system', 'timezone')

    def __init__(self, group, title=None, number=None, t_type=None, station_code_system=None, timezone=None,
                 vehicle=None, carrier=None, fare=None, sales=None, use_supplier_title=None):

        self.group = group
        self.title = title
        self.number = number
        self.vehicle = vehicle
        self.carrier = carrier
        self.fare = fare
        self.sales = sales

        self._schedules = []
        self._stoppoints = []
        self._raw_data = None

        self.timezone = timezone or group.timezone
        self.t_type = t_type or group.t_type
        self.station_code_system = station_code_system or group.station_code_system

        self.use_supplier_title = use_supplier_title

    def get_element(self):
        thread_el = etree.Element('thread')
        for attr in self.optional_attributes:
            if getattr(self, attr) is not None:
                thread_el.set(attr, getattr(self, attr))

        for attr in self.optional_inheritable_attributes:
            if getattr(self, attr) != getattr(self.group, attr, None):
                thread_el.set(attr, getattr(self, attr))

        if self.carrier:
            thread_el.set('carrier_code', self.carrier.code)

            if self.carrier.title:
                thread_el.set('carrier_title', self.carrier.title)

            if self.carrier.code_system != self.group.carrier_code_system:
                thread_el.set('carrier_code_system', self.carrier.code_system)

        if self.vehicle:
            thread_el.set('vehicle_code', self.vehicle.code)

            if self.vehicle.title:
                thread_el.set('vehicle_title', self.vehicle.title)

            if self.vehicle.code_system != self.group.vehicle_code_system:
                thread_el.set('vehicle_code_system', self.vehicle.code_system)

        if self.fare:
            thread_el.set('fare_code', self.fare.code)

        if self.sales is not None:
            thread_el.set('sales', u'1' if self.sales else u'0')

        if self._schedules:
            schedules_el = etree.SubElement(thread_el, 'schedules')
            for schedule in self._schedules:
                schedules_el.append(schedule.get_element())

        if self._stoppoints:
            stoppoints_el = etree.SubElement(thread_el, 'stoppoints')
            for stoppoint in self._stoppoints:
                stoppoints_el.append(stoppoint.get_element())

        if self._raw_data:
            raw_el = etree.SubElement(thread_el, 'raw')
            raw_el.text = base64.b64encode(zlib.compress(smart_str(self._raw_data, 'utf8')))

        if self.use_supplier_title:
            thread_el.set('_use_supplier_title', u'1')

        return thread_el

    def add_stoppoint_block(self, stoppoint):
        self._stoppoints.append(stoppoint)

    def add_schedule_block(self, schedule):
        self._schedules.append(schedule)

    def set_fare_block(self, fare):
        self.fare = fare

    def set_raw_data(self, raw_data):
        self._raw_data = raw_data

    @property
    def stoppoints(self):
        return list(self._stoppoints)


class ScheduleBlock(BaseBlock):
    optional_attributes = ('exclude_days', 'times', 'period_start_date', 'period_end_date', 'period_start_time',
                           'period_end_time', 'canceled')

    def __init__(self, thread, days, times=None, exclude_days=None, period_start_date=None, period_end_date=None,
                 period_start_time=None, period_end_time=None, canceled=None):

        self.thread = thread

        self.days = days
        self.exclude_days = exclude_days
        self.times = times

        self.period_start_date = period_start_date
        self.period_end_date = period_end_date
        self.period_start_time = period_start_time
        self.period_end_time = period_end_time

        self.canceled = canceled

    def get_element(self):
        schedule_el = etree.Element('schedule')

        schedule_el.set('days', self.days)

        for attr in self.optional_attributes:
            if getattr(self, attr) is not None:
                schedule_el.set(attr, getattr(self, attr))

        return schedule_el


class StoppointBlock(BaseBlock):
    arrival_shift = None
    departure_shift = None
    arrival_time = None
    departure_time = None
    arrival_day_shift = None
    departure_day_shift = None
    distance = None
    is_combined = None

    is_searchable_to = None
    is_searchable_from = None
    in_station_schedule = None
    in_thread = None

    timezone = None

    optional_attributes = (
        'arrival_shift', 'departure_shift',
        'arrival_time', 'departure_time',
        'arrival_day_shift', 'departure_day_shift',
        'distance',
        'is_nonstop', 'is_technical', 'is_fuzzy', 'is_combined'
    )
    internal_attributes = ('is_searchable_to', 'is_searchable_from',
                           'in_station_schedule', 'in_thread')
    optional_inheritable_attributes = ('timezone',)

    def __init__(self, thread, station, **kwargs):

        self.thread = thread
        self.station = station

        for attr in self.optional_attributes:
            setattr(self, attr, kwargs.pop(attr, None))

        for attr in self.optional_inheritable_attributes:
            setattr(self, attr, kwargs.pop(attr, getattr(thread, attr)))

        for attr in self.internal_attributes:
            setattr(self, attr, kwargs.pop(attr, None))

        if kwargs:
            raise Exception('StoppointBlock receive extra attributes %r' % kwargs)

    def get_element(self):
        stoppoint_el = etree.Element('stoppoint')

        stoppoint_el.set('station_title', self.station.title)
        stoppoint_el.set('station_code', self.station.code)

        if self.thread.station_code_system != self.station.code_system:
            stoppoint_el.set('station_code_system', self.station.code_system)

        for attr in self.optional_attributes:
            if getattr(self, attr) is not None:
                stoppoint_el.set(attr, unicode(getattr(self, attr)))

        for attr in self.optional_inheritable_attributes:
            if getattr(self, attr) is not None and getattr(self, attr) != getattr(self.thread, attr):
                stoppoint_el.set(attr, unicode(getattr(self, attr)))

        for attr in self.internal_attributes:
            if getattr(self, attr) is not None:
                stoppoint_el.set('_' + attr, unicode(getattr(self, attr)))

        return stoppoint_el


class FareBlock(BaseBlock):
    def __init__(self, group, code):
        self.group = group
        self.code = code

        self._prices = []

    def add_price_block(self, price_value, currency, stop_from, stop_to, **kwargs):
        price = PriceBlock(fare=self, price_value=price_value, currency=currency,
                           stop_from=stop_from, stop_to=stop_to, **kwargs)
        self._prices.append(price)

        return self._prices[-1]

    def get_element(self):
        fare_el = etree.Element('fare')
        fare_el.set('code', self.code)

        if self._prices:
            for price in self._prices:
                fare_el.append(price.get_element())

        return fare_el

    @property
    def key(self):
        return u"{}".format(self.code)


class PriceBlock(BaseBlock):
    def __init__(self, fare, price_value, currency, stop_from, stop_to,
                 data=None, is_min_price=None, oneway_fare=False):
        self.fare = fare
        self.price_value = price_value
        self.currency = currency
        self.stop_from = stop_from
        self.stop_to = stop_to
        self.is_min_price = is_min_price
        self.oneway_fare = oneway_fare
        self.data = data

    def get_element(self):
        price_el = etree.Element('price')
        price_el.set('price', self.price_value)
        price_el.set('currency', self.currency)

        if self.is_min_price is not None:
            price_el.set('is_min_price', self.is_min_price)

        if self.oneway_fare:
            price_el.set('oneway_fare', '1')

        stop_from_el = etree.SubElement(price_el, 'stop_from')
        stop_from_el.set('station_code', self.stop_from.code)
        stop_from_el.set('station_title', self.stop_from.title)

        if self.stop_from.code_system:
            stop_from_el.set('station_code_system', self.stop_from.code_system)

        stop_to_el = etree.SubElement(price_el, 'stop_to')
        stop_to_el.set('station_code', self.stop_to.code)
        stop_to_el.set('station_title', self.stop_to.title)

        if self.stop_to.code_system:
            stop_to_el.set('station_code_system', self.stop_to.code_system)

        if self.data:
            data_el = etree.SubElement(price_el, 'data')
            data_el.text = self.data

        return price_el
