# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import re
from datetime import datetime, timedelta, time
from dateutil import parser

from django.conf import settings

from common.models.geo import Station, Settlement, StationMajority
from common.models.schedule import Company
from common.models.transport import TransportModel, TransportType
from travel.rasp.library.python.common23.date import environment
from common.utils.calendar_matcher import get_matcher

from travel.rasp.library.python.common23.date.date import get_pytz
from travel.rasp.library.python.common23.date.run_mask import RunMask
from travel.rasp.library.python.common23.models.core.schedule.run_days import build_run_days

from common.utils.settlement import get_connected_stations


BARIS_TITLE_DASH = "\u2013"


class BarisData(object):
    """
    Данные из БАРиС по табло аэропорта и по запросам точка-точка за период и на все дни,
    дополненные информацией из базы Расписаний
    """
    def __init__(self, baris_response):
        self.flights = baris_response.flights
        self.stations_by_ids = Station.objects.in_bulk(baris_response.stations_ids)
        self.companies_by_ids = Company.objects.in_bulk(baris_response.companies_ids)
        self.transport_models_by_ids = TransportModel.objects.in_bulk(baris_response.transport_models_ids)

        settlements_keys = [
            'c{}'.format(station.settlement_id)
            for station in self.stations_by_ids.values()
            if station.settlement_id
        ]
        settlements_by_keys = Settlement.in_bulk(settlements_keys)
        self.settlements_by_ids = {settlement.id: settlement for settlement in settlements_by_keys.values()}


def get_plane_stations_ids(point):
    """
    Получение списка id станций для указанной точки
    """
    if isinstance(point, Station):
        return [point.id] if point.t_type.id == TransportType.PLANE_ID else []

    if isinstance(point, Settlement):
        stations = get_connected_stations(point, StationMajority.IN_TABLO_ID, TransportType.PLANE_ID)
        return [station.id for station in stations]

    return []


class BarisMasksProcessor(object):
    """
    Класс для получения информации о днях полета рейса
    """
    def __init__(self, baris_masks, departure_timezone):
        """
        :param baris_masks: маски из рейса БАРиС
        :param departure_timezone: таймзона для определения текущего времени
        """
        self.baris_masks = baris_masks
        self.pytz = get_pytz(departure_timezone)
        self.today = environment.now_aware().astimezone(self.pytz).date()
        self.from_date = self.today - timedelta(settings.SEARCH_DAYS_TO_PAST)
        self.to_date = self.today + timedelta(settings.DAYS_TO_FUTURE)

        days = set()
        self.nearest_run_date = None

        from_dt = datetime.combine(self.from_date, time(0))
        to_dt = datetime.combine(self.to_date, time(0))
        for mask in baris_masks:
            date_from = datetime.strptime(mask['from'], "%Y-%m-%d")
            date_until = datetime.strptime(mask['until'], "%Y-%m-%d")
            week_days = {int(char) for char in str(mask['on'])}

            dt = date_from
            while dt <= date_until:
                if from_dt <= dt <= to_dt and datetime.isoweekday(dt) in week_days:
                    days.add(dt)
                    if not self.nearest_run_date or dt.date() <= self.today:
                        self.nearest_run_date = dt

                dt += timedelta(1)

        self.run_mask = RunMask(today=self.today, days=days)

    def get_days_text(self, lang='ru', days_shift=0):
        """
        Получение строки дней хождения
        """
        run_mask = self.run_mask.shifted(shift=days_shift) if days_shift else self.run_mask

        matcher = get_matcher('basic', self.today, self.from_date, 365)
        template = matcher.find_template(run_mask, settings.DAYS_TO_FUTURE)

        return (
            template[0][lang]
            if template[0] is not None
            else unicode(run_mask.format_days_text(lang=lang))
        )

    def get_run_days(self, days_shift=0):
        """
        Получение списков дней хождения по месяцам
        """
        run_mask = self.run_mask.shifted(shift=days_shift) if days_shift else self.run_mask
        return build_run_days(run_mask, self.from_date)

    def get_nearest_datetime(self, time_text, day_shift=0, timezone=None):
        naive_dt = datetime.combine(
            self.nearest_run_date + timedelta(day_shift),
            parser.parse(time_text).time()
        )
        if not timezone:
            return naive_dt

        return get_pytz(timezone).localize(naive_dt)


def get_flight_title(baris_data, baris_flight, is_one_flight, separator=BARIS_TITLE_DASH):
    """Получение названия рейса в виде Город-Город"""
    if is_one_flight:  # Рейс
        first_station_id = baris_flight['route'][0]['airportID']
        last_station_id = baris_flight['route'][-1]['airportID']
    else:  # Поиск и табло
        first_station_id = baris_flight['route'][0]
        last_station_id = baris_flight['route'][-1]

    first_station = baris_data.stations_by_ids[first_station_id]
    last_station = baris_data.stations_by_ids[last_station_id]

    if first_station.settlement_id in baris_data.settlements_by_ids:
        first_point = baris_data.settlements_by_ids[first_station.settlement_id]
    else:
        first_point = first_station
    if last_station.settlement_id in baris_data.settlements_by_ids:
        last_point = baris_data.settlements_by_ids[last_station.settlement_id]
    else:
        last_point = last_station

    return '{} {} {}'.format(first_point.L_title(), separator, last_point.L_title())


def make_baris_masks_from_protobuf(schedule):
    """
    Формирование маски как списка словарей из списка объектов протобуфа (маски из TFlight.Schedule)
    """
    return [
        {'from': mask.From, 'until': mask.Until, 'on': mask.On}
        for mask in schedule.Masks
    ]


CYRILLIC_CHARACTERS_ENCODING = {
    'А': 'A', 'Б': 'B', 'В': 'V', 'Г': 'G', 'Д': 'D', 'Е': 'E', 'Ё': 'QO',
    'Ж': 'QZ', 'З': 'Z', 'И': 'I', 'Й': 'J', 'К': 'K', 'Л': 'L', 'М': 'M',
    'Н': 'N', 'О': 'O', 'П': 'P', 'Р': 'R', 'С': 'S', 'Т': 'T', 'У': 'U',
    'Ф': 'F', 'Х': 'H', 'Ц': 'C', 'Ч': 'QC', 'Ш': 'QH', 'Щ': 'QS',
    'Ъ': 'QD', 'Ы': 'QI', 'Ь': 'QT', 'Э': 'QE', 'Ю': 'QU', 'Я': 'QA'
}

SIMPLE_CHAR_DECODING = {
    'A': 'А', 'B': 'Б', 'V': 'В', 'G': 'Г', 'D': 'Д', 'E': 'Е', 'Z': 'З', 'I': 'И',
    'J': 'Й', 'K': 'К', 'L': 'Л', 'M': 'М', 'N': 'Н', 'O': 'О', 'P': 'П', 'R': 'Р',
    'S': 'С', 'T': 'Т', 'U': 'У', 'F': 'Ф', 'H': 'Х', 'C': 'Ц'
}

COMPLEX_CHAR_DECODING = {
    'O': 'Ё', 'Z': 'Ж', 'C': 'Ч', 'H': 'Ш', 'S': 'Щ', 'D': 'Ъ', 'I': 'Ы', 'T': 'Ь',
    'E': 'Э', 'U': 'Ю', 'A': 'Я'
}


def make_pseudo_uid_for_baris_flight(flight_number, airline_id, mask_last_date):
    """
    Генерация псевдо uid нитки рейса из БАРиС для пересадочника
    Русские буквы переводим в латиницу так, чтобы потом обратно можно было вытащить из uid номер рейса
    :param flight_number: Номер рейса
    :param airline_id: id аэрокомпании
    :param mask_last_date: последняя дата из маски дней хождения
    """
    has_cyrillic = False
    chars = []
    for char in flight_number:
        if char in CYRILLIC_CHARACTERS_ENCODING:
            chars.append(CYRILLIC_CHARACTERS_ENCODING[char])
            has_cyrillic = True
        elif char == ' ':
            chars.append('-')
        else:
            chars.append(char)
    number_for_uid = ''.join(chars)
    if has_cyrillic:
        number_for_uid = 'RUS{}'.format(number_for_uid)

    # Последняя дата хождения нитки, записанная в целое число (чтобы пересадочник думал, что это номер нитки)
    date_number = mask_last_date.strftime('%y%m%d')

    # 12 - код поставщика данных Official Airline Guide
    return u'{}_{}_c{}_12'.format(number_for_uid, date_number, airline_id)


class BarisPseudoUid(object):
    """
    Класс для разбора pseudo_uid нитки из БАРиС
    Выделяет номер рейса и дату последнего вылета из pseudo_uid
    """
    def __init__(self, thread_pseudo_uid):
        self.uid = thread_pseudo_uid
        self.is_baris_flight = bool(re.match(r'\w+[-]\w+_\d+_c\d+_\d+', thread_pseudo_uid))

        if self.is_baris_flight:
            uid_parts =  thread_pseudo_uid.split('_')
            try:
                self.flight_number = self._get_flight_number(uid_parts[0])
                self.last_date = self._get_last_date(uid_parts[1])
            except:
                self.is_baris_flight = False

        if not self.is_baris_flight:
            self.flight_number = None
            self.last_date = None

    def _get_flight_number(self, uid_flight_number):
        """
        Получаем номер рейса из псевдо-uid нитки из бариса
        Если номер рейса содержит русские буквы, то он начинается на 'RUS', и в этом случае декодируется
        """
        if uid_flight_number[:3] != 'RUS':
            return uid_flight_number

        chars = []
        index = 3
        while index < len(uid_flight_number):
            if uid_flight_number[index] == 'Q':
                index += 1
                chars.append(COMPLEX_CHAR_DECODING[uid_flight_number[index]])
            elif uid_flight_number[index] in SIMPLE_CHAR_DECODING:
                chars.append(SIMPLE_CHAR_DECODING[uid_flight_number[index]])
            else:
                chars.append(uid_flight_number[index])
            index += 1

        return ''.join(chars)

    def _get_last_date(self, uid_last_date):
        format = '%Y%m%d' if len(uid_last_date) == 8 else '%y%m%d'
        return datetime.strptime(uid_last_date, format).date()
