# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import pytz

import requests

from common.data_api.baris.helpers import BarisMasksProcessor

from travel.rasp.library.python.api_clients.baris import BarisClient


log = logging.getLogger(__name__)


class BarisService(object):
    """
    Клиент для единой базы авиарейсов (БАРиС)
    https://wiki.yandex-team.ru/raspisanija/avia/ya-storage/doc/api/
    """
    def __init__(
        self,
        baris_client  # type: BarisClient
    ):
        self._baris_client = baris_client

    def get_station_tablo(self, station_id, direction='departure', after=None, before=None, limit=None, terminal=None):
        """
        Табло аэропорта в рамках указанного времени
        :param station_id: id аэропорта
        :param direction: departure или arrival
        :param after: ограничение запришаваемого интервала снизу, передается без указания таймзоны
        :param before: ограничение запришаваемого интервала сверху, передается без указания таймзоны
        :param limit: ограничение на количество выдаваемых рейсов, если не задано, то количество не ограничено
        :param terminal: выдача только по указанному терминалу, если не задан, то выдаются рейсы по всему аэропорту
        :return: объект класса BarisResponse
        """

        try:
            baris_raw_data = self._baris_client.flight_board(station_id, direction, after, before, terminal, limit)
        except requests.HTTPError as ex:
            if ex.response.status_code == 404:
                return BarisResponse([], {station_id}, [], [])
            raise

        baris_response = self._make_baris_response(baris_raw_data)
        baris_response.stations_ids.add(station_id)

        for flight in baris_response.flights:
            if 'status' in flight and 'diverted' in flight['status'] and 'divertedAirportID' in flight['status']:
                diverted_station_id = flight['status']['divertedAirportID']
                if diverted_station_id:
                    baris_response.stations_ids.add(diverted_station_id)

        return baris_response

    def get_station_all_days_tablo(
        self, station_id, station_timezone, lang, direction='departure', result_timezone=None, run_date=None
    ):
        """
        Табло аэропорта на все дни
        :param station_id: id аэропорта
        :param station_timezone: таймзона аэропорта
        :param lang: код языка ('ru' и т.п.)
        :param direction: departure или arrival
        :param result_timezone: если указана, то в ответ добавляются дни хождения в этой таймзоне
        :param run_date: если указана, то выбираются только рейсы вылетающие в эту дату
        :return: объект класса BarisResponse
        """
        try:
            baris_raw_data = self._baris_client.flight_board_schedule(station_id, direction)
        except requests.HTTPError as ex:
            if ex.response.status_code == 404:
                return BarisResponse([], {station_id}, [], [])
            raise

        baris_response = self._make_baris_response(baris_raw_data)
        baris_response.stations_ids.add(station_id)

        result_pytz = pytz.timezone(result_timezone) if result_timezone else None

        result_flights = []
        for flight in baris_response.flights:
            for schedule in flight['schedules']:
                # Формируем рейсы в формате максимально похожем на табло одного дня
                processor = BarisMasksProcessor(schedule['masks'], station_timezone)
                if run_date and not processor.run_mask[run_date]:
                    continue

                days_text = processor.get_days_text(lang)
                if days_text and processor.nearest_run_date:
                    result_flight = {
                        'airlineID': flight['airlineID'],
                        'title': flight['title'],
                        'transportModelID': schedule.get('transportModelID', 0),
                        'terminal': schedule['terminal'],
                        'route': schedule['route'],
                        'time': schedule['time'],
                        'daysText': days_text,
                        'nearestDatetime': processor.get_nearest_datetime(schedule['time'], timezone=station_timezone),
                        'naiveStart': self._get_naive_start(schedule, processor)
                    }

                    # Если указана таймзона, то строку дней хождения формируем также и в этой таймзоне
                    if result_pytz:
                        tz_dt = result_flight['nearestDatetime'].astimezone(result_pytz)
                        days_shift = (tz_dt.date() - result_flight['nearestDatetime'].date()).days

                        if days_shift:
                            result_flight['tzDaysText'] = processor.get_days_text(lang, days_shift)
                        else:
                            result_flight['tzDaysText'] = result_flight['daysText']

                    result_flights.append(result_flight)

        baris_response.flights = result_flights

        return baris_response

    def get_p2p_search(
        self, station_from_ids, station_to_ids, after=None, before=None,
        national_version=None, show_banned=False, flight_numbers=None
    ):
        """
        Выдача поиска точка-точка в рамках указанного времени
        :param station_from_ids: список id станций отправления
        :param station_to_ids: список id станций прибытия
        :param after: ограничение запришаваемого интервала снизу, передается без указания таймзоны
        :param before: ограничение запришаваемого интервала сверху, передается без указания таймзоны
        :param national_version: код национальной версии ('RU' и т.п.)
        :param show_banned: показывать забаненые рейсы
        :param flight_numbers: список номеров рейсов, по которым фильтровать выдачу
        :return: объект класса BarisResponse
        """
        if not station_from_ids or not station_to_ids:
            return BarisResponse([], [], [], [])

        if flight_numbers is not None:
            baris_raw_data = self._baris_client.flight_p2p_with_numbers(
                station_from_ids=station_from_ids,
                station_to_ids=station_to_ids,
                national_version=national_version,
                show_banned=show_banned,
                flight_numbers=flight_numbers,
                before=before,
                after=after
            )
        else:
            baris_raw_data = self._baris_client.flight_p2p(
                station_from_ids=station_from_ids,
                station_to_ids=station_to_ids,
                national_version=national_version,
                show_banned=show_banned,
                before=before,
                after=after
            )
        baris_response = self._make_baris_response(baris_raw_data)
        baris_response.stations_ids |= set(station_from_ids) | set(station_to_ids)
        return baris_response

    def get_p2p_all_days_search(
        self, lang, station_from_ids, station_to_ids, from_timezone, to_timezone, national_version=None,
        show_banned=False, result_timezone=None, add_days_text=True, add_run_days=True
    ):
        """
        Выдача поиска точка-точка на все дни
        :param lang: код языка ('ru' и т.п.)
        :param station_from_ids - список id станций отправления
        :param station_to_ids - список id станций прибытия
        :param from_timezone - таймзона отправления, предполагается, одинаковой для всех станций отправления
        :param to_timezone - таймзона прибытия, предполагается, одинаковой для всех станций прибытия
        :param national_version - код национальной версии ('RU' и т.п.)
        :param show_banned - показывать забаненые рейсы
        :param result_timezone - если указана, то в ответ добавляются дни хождения в этой таймзоне
        :param add_days_text - вычислять для рейсов строки дней хождения
        :param add_run_days - вычислять для рейсов календарь дней хождения
        :return объект класса BarisResponse
        """
        if not station_from_ids or not station_to_ids or not from_timezone or not to_timezone:
            return BarisResponse([], [], [], [])

        baris_raw_data = self._baris_client.flight_p2p_schedule(
            station_from_ids=station_from_ids,
            station_to_ids=station_to_ids,
            national_version=national_version,
            show_banned=show_banned
        )
        baris_response = self._make_baris_response(baris_raw_data)
        baris_response.stations_ids |= set(station_from_ids) | set(station_to_ids)

        result_pytz = pytz.timezone(result_timezone) if result_timezone else None

        for flight in baris_response.flights:
            processor = BarisMasksProcessor(flight['masks'], from_timezone)

            if processor.nearest_run_date:
                flight['departure'] = processor.get_nearest_datetime(flight['departureTime'], timezone=from_timezone)
                flight['arrival'] = processor.get_nearest_datetime(
                    flight['arrivalTime'], flight['arrivalDayShift'], to_timezone
                )
                flight['naiveStart'] = self._get_naive_start(flight, processor)

                if add_days_text:
                    flight['daysText'] = processor.get_days_text(lang)
                if add_run_days:
                    flight['runDays'] = processor.get_run_days()

                # Если указана таймзона, то календарь и строку дней хождения формируем также и в этой таймзоне
                if result_pytz:
                    tz_departure = flight['departure'].astimezone(result_pytz)
                    days_shift = (tz_departure.date() - flight['departure'].date()).days

                    if add_days_text:
                        if days_shift:
                            flight['tzDaysText'] = processor.get_days_text(lang, days_shift)
                        else:
                            flight['tzDaysText'] = flight['daysText']

                    if add_run_days:
                        if days_shift:
                            flight['tzRunDays'] = processor.get_run_days(days_shift)
                        else:
                            flight['tzRunDays'] = flight['runDays']

        baris_response.flights = [flight for flight in baris_response.flights if 'departure' in flight]

        return baris_response

    def _get_naive_start(self, flight, processor):
        if 'startTime' in flight and flight['startTime'] and 'startDayShift' in flight:
            return processor.get_nearest_datetime(flight['startTime'], flight['startDayShift'])
        if 'time' in flight:
            return processor.get_nearest_datetime(flight['time'])
        return processor.get_nearest_datetime(flight['departureTime'])

    def get_flight_schedule(self, flight_number):
        """
        Информация о расписании рейса на все дни
        :param flight_number: номер рейса в формате типа SU-123
        """
        baris_raw_data = self._baris_client.flight_schedule(flight_number)
        return self._make_baris_response(baris_raw_data, one_flight=True)

    def _make_baris_response(self, baris_raw_data, one_flight=False):
        """
        Формирование базового ответа по данным из БАРиС
        Включает в себя формирование множеств id-шников станций, компаний и моделей самолетов
        """
        stations_ids = set()
        companies_ids = set()
        models_ids = set()

        flights = [baris_raw_data] if one_flight else baris_raw_data.get('flights') or []
        for flight in flights:
            self._process_baris_flight(flight, stations_ids, companies_ids, models_ids, one_flight)

        return BarisResponse(flights, stations_ids, companies_ids, models_ids)

    def _process_baris_schedule(self, schedule, stations_ids, models_ids, is_one_flight):
        for route_station in schedule['route']:
            if is_one_flight:
                stations_ids.add(route_station['airportID'])  # Рейс имеет другой формат поля route
            else:
                stations_ids.add(route_station)  # Табло и поиск

        if 'transportModelID' in schedule and schedule['transportModelID']:
            models_ids.add(schedule['transportModelID'])

    def _process_baris_flight(self, flight, stations_ids, companies_ids, models_ids, is_one_flight):
        companies_ids.add(flight['airlineID'])

        # Для выдачи точка-точка
        if 'arrivalStation' in flight:
            stations_ids.add(flight['arrivalStation'])
        if 'departureStation' in flight:
            stations_ids.add(flight['departureStation'])

        if 'schedules' in flight:
            # Для табло на все дни и рейса есть разбиение на schedules
            for schedule in flight['schedules']:
                self._process_baris_schedule(schedule, stations_ids, models_ids, is_one_flight)
        else:
            self._process_baris_schedule(flight, stations_ids, models_ids, is_one_flight)

            if 'codeshares' in flight and flight['codeshares']:
                for codeshare in flight['codeshares']:
                    companies_ids.add(codeshare['airlineID'])

    def get_delayed_flights(self, station_from_ids):
        """Получение информации о задержанных и отмененных рейсах для указанных аэропортов"""
        baris_raw_data = self._baris_client.delayed_flights(station_from_ids)
        return {
            station_data['id']: {
                'cancelled': station_data.get('cancelled', 0),
                'delayed': station_data.get('delayed', 0)
            }
            for station_data in baris_raw_data['stations']
        }

    def get_p2p_summary(self):
        """Получение пар всех станций в мире, между которыми есть авиасообщение"""
        baris_raw_data = self._baris_client.flight_p2p_summary()

        station_pairs = set()
        for flight in baris_raw_data['flights']:
            station_pairs.add((flight['departureStation'], flight['arrivalStation']))
        return station_pairs

    def get_station_summary(self):
        """Получение всех станций мире, для которых есть авиасообщение"""
        baris_raw_data = self._baris_client.flight_p2p_summary()

        departure_stations = set()
        arrival_stations = set()
        for flight in baris_raw_data['flights']:
            if flight['departureStation'] not in departure_stations:
                departure_stations.add(flight['departureStation'])
            if flight['arrivalStation'] not in arrival_stations:
                arrival_stations.add(flight['arrivalStation'])

        return departure_stations, arrival_stations


class BarisResponse(object):
    """
    Данные из БАРиС по табло аэропорта и поиску p2p на один день и на все дни
    """
    def __init__(self, flights, stations_ids, companies_ids, models_ids):
        self.flights = flights
        self.stations_ids = stations_ids
        self.companies_ids = companies_ids
        self.transport_models_ids = models_ids
        self.trusted = False
