# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

log = logging.getLogger(__name__)


class BaseOAuthApi(object):
    def __init__(self, token, base_url,
                 max_retries=5, backoff_factor=5.0, retry_status_list=(500, 502, 503, 504), timeout=5):

        self.base_url = base_url
        self._session = requests.Session()
        self._last_response = None
        self.timeout = timeout

        # Нужно при работе с sandbox в дев-режиме
        if token:
            self._session.headers.update({'Authorization': 'OAuth {}'.format(token)})

        # http://urllib3.readthedocs.io/en/latest/reference/urllib3.util.html#module-urllib3.util.retry
        # urllib3 will sleep for: {backoff factor} * (2 ^ ({number of total retries} - 1))
        # seconds. If the backoff_factor is 0.1, then sleep() will sleep for [0.0s, 0.2s, 0.4s, …] between retries.
        retries = Retry(total=max_retries, backoff_factor=backoff_factor, status_forcelist=retry_status_list)
        adapter = HTTPAdapter(max_retries=retries)
        self._session.mount('http://', adapter)
        self._session.mount('https://', adapter)

    def _process_text_query(self, method, url, *args, **kwargs):
        if 'timeout' not in kwargs:
            kwargs['timeout'] = self.timeout

        url = self.base_url + url
        self._last_response = getattr(self._session, method)(url, *args, **kwargs)
        try:
            self._last_response.raise_for_status()
        except requests.HTTPError:
            log.exception('Api Error')
            log.error('Response body: %s', self._last_response.text)
            raise

        return self._last_response.text

    def _process_json_query(self, method, url, *args, **kwargs):
        return json.loads(self._process_text_query(method, url, *args, **kwargs))
