# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

"""
Клиент для сервиса TRUST
https://wiki.yandex-team.ru/TRUST/Payments/

Пример создания платежа и отправки его на оплату:

    client = TrustClient(user_ip='127.0.0.1', user_region_id=54)

    order_id_1 = client.create_order('ticket')
    order_id_2 = client.create_order('ticket')

    order1 = TrustPaymentOrder(order_id=order_id_1, price=1000, fiscal_nds=FiscalNdsType.NDS_18, fiscal_title='order-1')
    order2 = TrustPaymentOrder(order_id=order_id_2, price=2000, fiscal_nds=FiscalNdsType.NDS_18, fiscal_title='order-2')

    purchase_token = client.create_payment(
        [order1, order2], 7200, 'RUB', False,
        user_email='lorekhov@yandex-team.ru',
        fiscal_data=TrustFiscalData(
            fiscal_taxation_type=FiscalTaxationType.OSN,
            fiscal_partner_inn=66777610,
            fiscal_partner_phone='+74342578903'
        )
    )
    payment_url = client.start_payment(purchase_token)

Пример получения чека:
    receipt_data = client.get_receipt(purchase_token, purchase_token)
"""

import json
import logging
from collections import namedtuple
from copy import copy
from decimal import Decimal

import requests
from django.conf import settings
from django.core.serializers.json import DjangoJSONEncoder
from django.utils.encoding import force_text
from enum import Enum

from common.data_api.billing import TRUST_TIMEOUT, ERROR_STATUS, SUCCESS_STATUS, TRUST_HOSTS_BY_ENVIRONMENT
from common.dynamic_settings.default import conf
from common.settings.configuration import Configuration
from common.utils.exceptions import SimpleUnicodeException
from common.utils.namedtuple import namedtuple_with_defaults
from common.utils.yasmutil import Metric, MeasurableDecorator

log = logging.getLogger(__name__)


class TrustReceiptException(SimpleUnicodeException):
    pass


class TrustClientException(SimpleUnicodeException):
    pass


class TrustClientRequestError(TrustClientException):
    pass


class TrustClientConnectionError(TrustClientRequestError):
    pass


class TrustClientHttpError(TrustClientRequestError):
    pass


class TrustClientInvalidStatus(TrustClientException):
    pass


class WrongFiscalNdsType(TrustClientException):
    pass


class ArgumentError(TrustClientException):
    pass


class TrustPaymentStatuses(Enum):
    NOT_STARTED = 'not_started'
    STARTED = 'started'
    AUTHORIZED = 'authorized'
    NOT_AUTHORIZED = 'not_authorized'
    CLEARED = 'cleared'
    CANCELED = 'canceled'
    REFUNDED = 'refunded'


def train_order_skip_trust():
    return (settings.APPLIED_CONFIG != Configuration.PRODUCTION
            and (settings.DEBUG_TRAIN_ORDER_SKIP_TRUST or conf.TRAIN_PURCHASE_SKIP_TRUST))


def get_payment_status_from_info(payment_info):
    payment_status = payment_info['payment_status']
    try:
        return TrustPaymentStatuses(payment_status)
    except ValueError:
        raise TrustClientInvalidStatus('Неизвестный статус корзины "{}".'.format(payment_status))


class TrustPaymentInfo(object):
    def __init__(self, payment_info):
        self.status = get_payment_status_from_info(payment_info)
        self._payment_info = payment_info

    @property
    def resp_code(self):
        return self._payment_info.get('payment_resp_code')

    @property
    def resp_desc(self):
        return self._payment_info.get('payment_resp_desc')

    @property
    def reversal_id(self):
        return self._payment_info.get('reversal_id')


class FiscalTaxationType(object):
    OSN = 'OSN'
    USN_INCOME = 'USN_income'
    USN_INCOME_MINUS_CHARGE = 'USN_income_minus_charge'
    ESN_CALC_INCOME = 'ESN_calc_income'
    ESN_AGRICULTURE = 'ESN_agriculture'
    PATENT = 'patent'


class FiscalNdsType(object):
    NDS_20 = 'nds_20'
    NDS_18 = 'nds_18'
    NDS_10 = 'nds_10'
    NDS_0 = 'nds_0'
    NDS_NONE = 'nds_none'
    NDS_20_120 = 'nds_20_120'
    NDS_18_118 = 'nds_18_118'
    NDS_10_110 = 'nds_10_110'

    @classmethod
    def from_rate(cls, rate):
        try:
            return RATE_TO_FISCAL_NDS_TYPE[rate if rate is None else rate.quantize(Decimal('.00'))]
        except KeyError:
            raise WrongFiscalNdsType('Неизвестная налоговая ставка {}'.format(rate))


RATE_TO_FISCAL_NDS_TYPE = {
    None: FiscalNdsType.NDS_NONE,
    Decimal('0'): FiscalNdsType.NDS_0,
    Decimal('9.09'): FiscalNdsType.NDS_10_110,
    Decimal('10'): FiscalNdsType.NDS_10,
    Decimal('15.25'): FiscalNdsType.NDS_18_118,
    Decimal('18'): FiscalNdsType.NDS_18,
    Decimal('16.67'): FiscalNdsType.NDS_20_120,
    Decimal('20'): FiscalNdsType.NDS_20,
}

EXCEPTION_NAME_TO_METRIC = {
    TrustClientConnectionError: 'connection_errors_cnt',
    TrustClientHttpError: 'http_errors_cnt',
    TrustClientInvalidStatus: 'invalid_status_errors_cnt',
}


class TrustRefundStatuses(Enum):
    WAIT_FOR_NOTIFICATION = 'wait_for_notification'
    SUCCESS = 'success'


TrustPaymentOrder = namedtuple_with_defaults('TrustPaymentOrder', (
    'order_id',
    'price',
    'fiscal_nds',
    'fiscal_title',
    'fiscal_inn',
))

TrustFiscalData = namedtuple('TrustFiscalData', ('fiscal_taxation_type', 'fiscal_partner_inn', 'fiscal_partner_phone'))

TrustRefundOrder = namedtuple('TrustRefundOrder', ('order_id', 'delta_amount'))


class measurable(MeasurableDecorator):
    """
    Декоратор, отправляющий в Голован метрики по всем запросу в траст.

    Собираемые метрики:
      trust.{endpoint}.timings_ahhh  - тайминги запросов
      trust.{endpoint}.connection_errors_cnt_ammm  - количество сетевых ошибок
      trust.{endpoint}.http_errors_cnt_ammm  - количество HTTP ошибок (код ответа отличен от 2xx)
      trust.{endpoint}.invalid_status_errors_cnt_ammm  - количество ошибок траста: невалидный статус
      trust.{endpoint}.errors_cnt_ammm  - общее количество ошибок
      trust.connection_errors_cnt_ammm  - кумулятивная метрика по количеству сетевых ошибок
      trust.http_errors_cnt_ammm  - кумулятивная метрика по количеству HTTP ошибок (код ответа отличен от 2xx)
      trust.invalid_status_errors_cnt_ammm - кумулятивная метрика по количеству невалидных статусов
      trust.errors_cnt_ammm  - общее количество ошибок по всем ручкам
    """
    prefix = 'trust'

    def _handle_error(self, exc):
        result = super(measurable, self)._handle_error(exc)
        exc_type = type(exc)
        if exc_type in EXCEPTION_NAME_TO_METRIC:
            metric_name = EXCEPTION_NAME_TO_METRIC[type(exc)]
            result.extend([
                Metric(self._name('errors_cnt'), 1, 'ammm'),
                Metric(self._name(metric_name), 1, 'ammm'),
                Metric('errors_cnt', 1, 'ammm'),
                Metric(metric_name, 1, 'ammm')
            ])
            return result
        return result


class TrustClient(object):
    def __init__(self, trust_host=None, user_ip=None, user_passport_uid=None, user_region_id=None):
        self.trust_url = self._build_trust_url(trust_host)
        self.headers = {k: force_text(v) for k, v in {
            'X-User-Ip': user_ip,
            'X-Uid': user_passport_uid,
            'X-Region-Id': user_region_id
        }.items() if v}
        self.headers['X-Service-Token'] = settings.TRUST_SERVICE_TOKEN

    @measurable()
    def create_partner(self, partner_data=None):
        """
        Создает партнера, наш партнер должен быть уже создан.
        Поэтому данный метод нужен только в деве и тестинге.
        """
        if getattr(settings, 'APPLIED_CONFIG', None) == 'production':
            raise TrustClientException('Нельзя создавать партнера в проде.')

        url = self.trust_url + 'partners'
        partner_data = partner_data or {'city': 'Moscow', 'name': 'УФС-тест', 'email': 'ufs-test@example.com'}
        headers = {'X-Operator-Uid': '4005302934'}
        result = self._post(url, data=partner_data, headers=headers, error_message='Не удалось создать партнера.')
        return result['partner_id']

    @measurable()
    def is_partner_exist(self, partner_id):
        url = self.trust_url + 'partners/{}'.format(partner_id)
        try:
            self._get(url, error_message='Не удалось узнать статус партнера.')
        except TrustClientInvalidStatus:
            return False
        else:
            return True

    @measurable()
    def create_product(self, partner_id, product_name, product_id, is_service_fee=False, service_fee=None):
        """
        Создает наш продукт. Поэтому id мы должны задать сами.
        :param partner_id: ID партнера
        :param product_name: наименование продукта - строка
        :param product_id: ID продукта, может быть и строкой
        :param is_service_fee: (deprecated) является ли продукт комиссией сервиса
        :param service_fee: код комиссии продукта
        :return: ID продукта
        """
        url = self.trust_url + 'products'
        if service_fee is None:
            service_fee = 1 if is_service_fee else 0
        product_data = {
            'name': product_name,
            'partner_id': partner_id,
            'product_id': product_id,
            'service_fee': service_fee,
        }
        self._post(url, data=product_data, error_message='Не удалось создать продукт.')
        return product_id

    @measurable()
    def is_product_exist(self, product_id):
        url = self.trust_url + 'products/{}'.format(product_id)
        try:
            self._get(url, error_message='Не удалось узнать статус продукта.')
        except TrustClientInvalidStatus:
            return False
        else:
            return True

    @measurable()
    def create_order(self, product_id):
        url = self.trust_url + 'orders'
        order_data = {'product_id': product_id}
        result = self._post(url, data=order_data, error_message='Не удалось создать заказ.')
        return result['order_id']

    @measurable()
    def create_payment(
            self, payment_orders, payment_timeout, currency='RUB', is_mobile=False, user_email=None, fiscal_data=None,
            domain_sfx='ru', fiscal_title=None, return_path=None, pass_params=None,
    ):
        """
        :type payment_orders: list of TrustPaymentOrder
        :type payment_timeout: float, int, long
        :type currency: basestring
        :type is_mobile: bool
        :type user_email: str
        :type fiscal_data: TrustFiscalData
        :type domain_sfx: basestring
        :type fiscal_title: basestring
        :type return_path: str
        :type pass_params: dict
        """
        url = self.trust_url + 'payments'
        payment_data = {
            'currency': currency,
            'payment_timeout': payment_timeout,
            'orders': [o._asdict() for o in payment_orders],
            'template_tag': 'mobile/form' if is_mobile else 'desktop/form',
            'domain_sfx': domain_sfx
        }
        if user_email:
            payment_data['user_email'] = user_email
        if fiscal_data:
            payment_data.update(fiscal_data._asdict())
        if fiscal_title:
            payment_data['fiscal_title'] = fiscal_title
        if return_path:
            payment_data['return_path'] = return_path
        if pass_params:
            payment_data['pass_params'] = pass_params

        result = self._post(url, data=payment_data, error_message='Не удалось создать корзину.')
        return result['purchase_token']

    @measurable()
    def start_payment(self, purchase_token):
        url = self.trust_url + 'payments/{}/start'.format(purchase_token)
        result = self._post(url, error_message='Не смогли запустить корзину на оплату.')
        return result['payment_url']

    def get_raw_payment_info(self, purchase_token, error_message='Не удалось узнать информацию о корзине.'):
        url = self.trust_url + 'payments/{}'.format(purchase_token)
        return self._get(url, error_message=error_message, params={'show_trust_payment_id': 1})

    @measurable()
    def get_payment_info(self, purchase_token):
        return TrustPaymentInfo(self.get_raw_payment_info(purchase_token))

    @measurable()
    def is_payment_cleared(self, purchase_token):
        payment_info = self.get_raw_payment_info(purchase_token)
        return bool(payment_info.get('clear_real_ts'))

    @measurable()
    def get_payment_status(self, purchase_token):
        error_message = 'Не удалось узнать статус корзины.'
        payment_info = self.get_raw_payment_info(purchase_token, error_message=error_message)
        return get_payment_status_from_info(payment_info)

    @measurable()
    def clear_payment(self, purchase_token):
        url = self.trust_url + 'payments/{}/clear'.format(purchase_token)
        self._post(url, error_message='Не смогли сделать клиринг.')

    @measurable()
    def resize_payment_order(self, purchase_token, order_id, amount):
        url = self.trust_url + 'payments/{}/orders/{}/resize'.format(purchase_token, order_id)
        self._post(url, error_message='Не смогли сделать ресайз.', data={'amount': amount, 'qty': 0, 'data': None})

    @measurable()
    def clear_payment_order(self, purchase_token, order_id):
        url = self.trust_url + 'payments/{}/orders/{}/clear'.format(purchase_token, order_id)
        self._post(url, error_message='Не смогли сделать построчный клиринг.')

    @measurable()
    def unhold_payment_order(self, purchase_token, order_id):
        url = self.trust_url + 'payments/{}/orders/{}/unhold'.format(purchase_token, order_id)
        self._post(url, error_message='Не смогли сделать отмену строки.')

    @measurable()
    def unhold_payment(self, purchase_token):
        url = self.trust_url + 'payments/{}/unhold'.format(purchase_token)
        self._post(url, error_message='Не смогли сделать отмену.')

    @measurable()
    def create_refund(self, refund_orders, purchase_token,
                      reason_description='Refund', currency='RUB'):
        """
        :type refund_orders: list of TrustRefundOrder
        :type purchase_token: basestring
        :type reason_description: basestring
        :type currency: basestring
        """
        url = self.trust_url + 'refunds'
        refund_data = {
            'purchase_token': purchase_token,
            'orders': [o._asdict() for o in refund_orders],
            'currency': currency,
            'reason_desc': reason_description,
        }
        result = self._post(url, data=refund_data, error_message='Не удалось создать возврат.')
        return result['trust_refund_id']

    @measurable()
    def start_refund(self, trust_refund_id):
        url = self.trust_url + 'refunds/{}/start'.format(trust_refund_id)
        self._post(url, error_message='Не смогли сделать возврат.', check_for_success_status=False)

    @measurable()
    def get_refund_info(self, trust_refund_id, error_message='Не удалось узнать информацию о возврате.'):
        url = self.trust_url + 'refunds/{}'.format(trust_refund_id)
        return self._get(url, error_message=error_message, check_for_success_status=False)

    @measurable()
    def get_refund_status(self, trust_refund_id):
        error_message = 'Не удалось узнать статус возврата.'
        result = self.get_refund_info(trust_refund_id, error_message=error_message)
        refund_status = result['status']
        try:
            return TrustRefundStatuses(refund_status)
        except ValueError:
            raise TrustClientInvalidStatus('{} Неизвестный статус возврата "{}".'.format(error_message, refund_status))

    @measurable()
    def get_receipt_url(self, purchase_token):
        info = self.get_raw_payment_info(purchase_token)
        return info.get('fiscal_receipt_url')

    @measurable()
    def get_receipt_clearing_url(self, purchase_token):
        info = self.get_raw_payment_info(purchase_token)
        return info.get('fiscal_receipt_clearing_url')

    def get_refund_receipt_url(self, refund_id):
        info = self.get_refund_info(refund_id)
        return info.get('fiscal_receipt_url')

    @measurable()
    def get_receipt_pdf(self, purchase_token, refund_id=None, resize=False):
        """
        :param purchase_token: purchase_token покупки
        :param refund_id: None для чека на покупку
                          refund_id для чека на возврат
        :param resize: запрос ссылки на чек после ресайза
        :return: dict, описывающий чек
        """
        if refund_id and resize:
            raise ArgumentError('refund_id и resize=True недопустимы одновременно')

        if refund_id:
            url = self.get_refund_receipt_url(refund_id)
        elif resize:
            url = self.get_receipt_clearing_url(purchase_token)
        else:
            url = self.get_receipt_url(purchase_token)

        if url:
            url = url.replace('mode=mobile', 'mode=pdf')

        try:
            response = requests.get(
                url,
                timeout=TRUST_TIMEOUT,
                verify=self._get_verify()
            )
            response.raise_for_status()
            return response.content
        except Exception:
            message = 'Ошибка при рендеринге чека, url: {0}'.format(url)
            log.exception(message)
            raise TrustReceiptException(message)

    def _post(self, url, error_message, data=None, headers=None, check_for_success_status=True):
        _headers = copy(headers)
        if _headers is None:
            _headers = {}
        _headers.update(self.headers)
        return self._retrieve_and_check(
            error_message=error_message,
            request_function=requests.post,
            url=url,
            request_kwargs={'json': data, 'headers': _headers, 'timeout': TRUST_TIMEOUT, 'verify': self._get_verify()},
            check_for_success_status=check_for_success_status
        )

    def _get(self, url, error_message, check_for_success_status=True, params=None):
        return self._retrieve_and_check(
            error_message=error_message,
            request_function=requests.get,
            url=url,
            request_kwargs={
                'headers': self.headers,
                'timeout': TRUST_TIMEOUT,
                'verify': self._get_verify(),
                'params': params or {},
            },
            check_for_success_status=check_for_success_status
        )

    @staticmethod
    def _get_verify():
        return conf.TRUST_USE_PRODUCTION_FOR_TESTING or settings.TRUST_VERIFY

    @staticmethod
    def _retrieve_and_check(request_function, url, request_kwargs, error_message, check_for_success_status=True):
        try:
            data = request_kwargs.get('json')
            if data:
                log.info('Start request {url} with data\n{data}'.format(
                    url=url,
                    data=force_text(json.dumps(
                        data, indent=2, ensure_ascii=False, sort_keys=True, cls=DjangoJSONEncoder))
                ))
            response = request_function(url, **request_kwargs)
            log.info('Done request {url} with status {status} and content\n{content}'.format(
                url=response.request.url,
                status=response.status_code,
                content=force_text(json.dumps(response.json(), indent=2, ensure_ascii=False, sort_keys=True))
            ))
        except Exception:
            log.exception(error_message)
            raise TrustClientConnectionError(error_message)

        try:
            response.raise_for_status()
        except requests.HTTPError as ex:
            log.exception('Error occurred %s\n for request with body\n %s', error_message, response.request.body)
            if 500 <= ex.response.status_code < 600:
                # 5xx код - это шанс что повторный запрос исправит ситуацию
                raise TrustClientHttpError(error_message)
            raise TrustClientInvalidStatus(error_message)

        result = response.json()

        if result.get('status') == ERROR_STATUS:
            raise TrustClientInvalidStatus('{} Error code: "{}", error: "{}".'.format(
                error_message, result.get('status_code'), result.get('status_desc')))

        if check_for_success_status and result['status'] != SUCCESS_STATUS:
            raise TrustClientInvalidStatus(error_message)

        return result

    @staticmethod
    def _build_trust_url(host):
        if host is None:
            if conf.TRUST_USE_PRODUCTION_FOR_TESTING:
                host = TRUST_HOSTS_BY_ENVIRONMENT[Configuration.PRODUCTION]
            else:
                host = settings.TRUST_HOST

        return 'https://{host}:{port}/trust-payments/v2/'.format(host=host, port=settings.TRUST_PORT)
