# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os

from django.core.files.base import File

from common.db.mds.base import MDSS3Wrapper
from common.data_api.sandbox.sandbox_public_api import sandbox_public_api
from common.utils.download_file import download_file


log = logging.getLogger(__name__)


class WrapperType(object):
    FILE = None
    MDS = 'MDS'
    SANDBOX = 'SANDBOX'
    LOCAL_FILE = 'LOCAL_FILE'


class FileWrapper(File):
    """
    Обертка для файла - https://st.yandex-team.ru/RASPFRONT-8874
    Реальный объект может быть файлом на диске/mds/sandbox
    """
    type = WrapperType.FILE

    def __init__(self, path, file=None):
        self.path = path
        super(FileWrapper, self).__init__(file, path)

    def is_file_exist(self):
        return self.path and os.path.exists(self.path)

    def is_file_data_exist(self):
        try:
            self.load_from_storage()
        except Exception:
            return False

        return self.is_file_exist()

    def load_from_storage(self):
        raise NotImplementedError

    def download(self):
        self.load_from_storage()

    def upload(self):
        raise NotImplementedError

    def delete(self):
        if os.path.exists(self.path):
            os.remove(self.path)

    def open(self, mode='r'):
        if mode == 'w' and not os.path.exists(self.path):
            self.file = open(self.path, mode=mode)
        else:
            self.load_from_storage()

        super(FileWrapper, self).open(mode)
        return self

    def __str__(self):
        return '{}_{}'.format(self.type, self.path)


class MdsFileWrapper(FileWrapper):
    type = WrapperType.MDS

    def __init__(self, path, bucket=None, key=None, mds_client=None):
        self.bucket = bucket
        self.key = key if key is not None else path
        self.mds_client = mds_client or MDSS3Wrapper(bucket=self.bucket)

        super(MdsFileWrapper, self).__init__(path)

    def download_from_mds(self):
        self.mds_client.client.download_file(self.bucket, self.key, self.path)
        log.info('download_from_mds {} {} {}'.format(self.bucket, self.key, self.path))

    def load_from_storage(self):
        try:
            if self.is_file_exist():
                return
            else:
                self.download_from_mds()
        except Exception:
            log.exception('mds load_from_storage fail')
            raise

    def upload(self):
        self.mds_client.client.upload_file(Filename=self.path, Bucket=self.bucket, Key=self.key)

    def delete(self):
        self.mds_client.client.delete_object(Bucket=self.bucket, Key=self.key)
        super(MdsFileWrapper, self).delete()


class SandboxFileWrapper(FileWrapper):
    type = WrapperType.SANDBOX
    download_timeout = 60 * 20

    def __init__(self, path,  resource_type, attrs=None, ttl='inf'):
        self.resource_type = resource_type
        self.attrs = attrs or {}
        self.ttl = ttl

        super(SandboxFileWrapper, self).__init__(path)

    def download_from_sandbox(self):
        resource = sandbox_public_api.get_resource(resource_type=self.resource_type, attrs=self.attrs)
        url = resource['http']['proxy']
        download_file(url=url, filepath=self.path, timeout=self.download_timeout)
        log.info('download_from_sandbox {} {}'.format(url, self.path))

    def upload(self):
        sandbox_public_api.upload_file(self.resource_type, self.path, ttl=self.ttl, attrs=self.attrs)

    def load_from_storage(self):
        try:
            if self.is_file_exist():
                return
            else:
                self.download_from_sandbox()
        except Exception:
            log.exception('sandbox load_from_storage fail')
            raise


class LocalFileWrapper(FileWrapper):
    type = WrapperType.LOCAL_FILE

    def __init__(self, path, *args, **kwargs):
        super(LocalFileWrapper, self).__init__(path)

    def check_file(self):
        if self.is_file_exist() is False:
            raise Exception('file not found {}'.format(self.path))

    def upload(self):
        self.check_file()

    def load_from_storage(self):
        self.check_file()


class DirectoryWrapper(object):
    type = WrapperType.FILE

    def __init__(self, directory, *args, **kwargs):
        self.dir = directory

    def download(self):
        raise NotImplementedError

    def upload(self):
        raise NotImplementedError

    def __str__(self):
        return '{}_{}'.format(self.type, self.dir)


class MdsDirectoryWrapper(DirectoryWrapper):
    type = WrapperType.MDS

    def __init__(self, directory, mds_client, download_params=None, upload_params=None, delete_params=None):
        self.mds_client = mds_client
        self.download_params = download_params or {}
        self.upload_params = upload_params or {}
        self.delete_params = delete_params or {}

        super(MdsDirectoryWrapper, self).__init__(directory)

    def download(self):
        self.mds_client.download_directory(
            directory_path=self.dir,
            **self.download_params
        )

    def upload(self):
        self.mds_client.upload_directory(
            dirpath=self.dir,
            **self.upload_params
        )

    def delete(self):
        self.mds_client.delete_prefix_keys(
            **self.delete_params
        )


class LocalDirectoryWrapper(DirectoryWrapper):
    type = WrapperType.LOCAL_FILE

    def __init__(self, directory, *args, **kwargs):
        super(LocalDirectoryWrapper, self).__init__(directory)

    def upload(self):
        log.info('upload directory: {}'.format(self.dir))

    def download(self):
        log.info('download directory: {}'.format(self.dir))

    def delete(self):
        log.info('delete directory: {}'.format(self.dir))
