# coding: utf-8
from datetime import datetime, timedelta

from django.conf import settings
from pybreaker import CircuitBreaker

from common.db.mongo import database
from common.db.mongo.bulk_buffer import BulkBuffer
from common.models.currency import Price
from common.models.geo import Settlement, BaseStation
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date.environment import now_aware
from travel.library.python.tracing.instrumentation import traced_function


class MinPriceError(Exception):
    pass


define_setting('MIN_PRICES_BREAKER_PARAMS', default={'fail_max': 3, 'reset_timeout': 60})

min_prices_breaker = CircuitBreaker(**settings.MIN_PRICES_BREAKER_PARAMS)


def _get_point_type(point):
    if isinstance(point, Settlement):
        return 'Settlement'
    if isinstance(point, BaseStation):
        return 'Station'

    raise MinPriceError('Point {} has wrong type'.format(point))


class MinPriceStorage(object):
    indexes = {
        # индекс, необходимый для сохранения
        'main_for_save': ['route_uid', 'date_forward', 'object_from_id', 'object_to_id', 'class', 'key'],

        # индексы, необходимые для удаления
        'remove_by_date_forward': ['date_forward'],
        'remove_by_timestamp': ['timestamp'],

        # индекс, необходимый для чтения
        'read_by_settlement_settlement': ['Settlement_from_id', 'Settlement_to_id', 'type', 'class', 'route_uid'],
        'read_by_settlement_station': ['Settlement_from_id', 'Station_to_id', 'type', 'class', 'route_uid'],
        'read_by_station_settlement': ['Station_from_id', 'Settlement_to_id', 'type', 'class', 'route_uid'],
        'read_by_station_station': ['Station_from_id', 'Station_to_id', 'type', 'class', 'route_uid'],
    }

    def __init__(self, collection=None):
        self._collection = collection

    @property
    def collection(self):
        if not self._collection:
            self._collection = database.min_prices
        return self._collection

    @min_prices_breaker
    @traced_function
    def get_min_prices(self, point_from, point_to, t_codes=None, date_from=None):
        """
        Возвращает список минимальных цен, сгруппированных по ключам
        Например:

        {
            'route_id': 'A001',
            'class': 'economy',
            'type': 'train',
            'price': 1001.01,
        }
        :param point_from: Точка от которой ищем
        :type point_from: Point
        :param point_to: Точка до которой ищем
        :type point_to: Point
        :param t_codes: Список транспортных кодов, например ['bus', 'train']
        :type t_codes: list
        :param date_from: Дата, от которой начинаем искать (если не указана, будет взята текущая)
        :return: список минимальных цен, сгруппированных по ключам
        """
        if date_from is None:
            date_from = now_aware().date()

        match_query = {
            '{}_from_id'.format(_get_point_type(point_from)): point_from.id,
            '{}_to_id'.format(_get_point_type(point_to)): point_to.id,
            'date_forward': {'$gte': date_from.strftime('%Y-%m-%d')}
        }

        if t_codes:
            match_query['type'] = {'$in': t_codes}

        prices = self.collection.aggregate([
            {
                '$match': match_query
            },
            {
                '$group': {
                    '_id': {
                        'type': '$type',
                        'route_uid': '$route_uid',
                        'class': '$class'
                    },
                    'price': {'$min': '$price'}
                }
            },
            {
                '$project': {
                    '_id': 0,
                    'route_uid': '$_id.route_uid',
                    'class': '$_id.class',
                    'type': '$_id.type',
                    'price': 1
                }
            }
        ])
        return list(prices)

    @min_prices_breaker
    @traced_function
    def find_best_offers(self, t_type, departure_settlement, arrival_settlements, date_from=None):
        if date_from is None:
            date_from = now_aware().date()

        id_to_arrival_settlement = {
            arrival_settlement.id: arrival_settlement for arrival_settlement in arrival_settlements
        }
        documents = self.collection.aggregate([
            {
                '$match': {
                    'type': t_type.code,
                    'date_forward': {'$gte': date_from.strftime('%Y-%m-%d')},
                    'Settlement_from_id': departure_settlement.id,
                    'Settlement_to_id': {'$in': id_to_arrival_settlement.keys()}
                }
            },
            {
                '$sort': {'price': 1}
            },
            {
                '$group': {
                    '_id': '$Settlement_to_id',
                    'date_forward': {'$first': '$date_forward'},
                    'price': {'$first': '$price'},
                    'route_uid': {'$first': '$route_uid'}
                }
            }
        ])

        return {
            id_to_arrival_settlement[document['_id']]: {
                'departure_date': datetime.strptime(document['date_forward'], '%Y-%m-%d').date(),
                'number': document['route_uid'],
                'price': Price(document['price'])
            }
            for document in documents
        }

    def save_many(self, min_prices):
        """
        Сохраняет цены в mongodb.
        Расчитывает, что новая порция данных приходит с большими timestamp, чем старая.
        Цены пишутся поверх старых без проверки условий.
        :param min_prices: генератор строк с минимальными ценами, обязаны быть поля из key_fields
        :return:
        """
        self.create_indexes()

        key_fields = ['route_uid', 'date_forward', 'object_from_id', 'object_to_id', 'class', 'key']
        with BulkBuffer(self.collection, max_buffer_size=100000) as coll_buff:
            for row in min_prices:
                row['price'] = float(row['price'])
                row['{}_from_id'.format(row['object_from_type'])] = int(
                    row['{}_from_id'.format(row['object_from_type'])]
                )
                row['{}_to_id'.format(row['object_to_type'])] = int(row['{}_to_id'.format(row['object_to_type'])])
                row['object_from_id'] = int(row['object_from_id'])
                row['object_to_id'] = int(row['object_to_id'])

                keys = {field: row[field] for field in key_fields if field in row}
                values = {key: value for (key, value) in row.items() if key not in key_fields}
                coll_buff.update(keys, {'$set': values}, upsert=True)

    def create_indexes(self):
        for name, index in self.indexes.items():
            index_spec = [(field_name, 1) for field_name in index]
            self.collection.create_index(index_spec, name=name, background=True)

    def remove_old_rows(self, days_ago=3):
        now = now_aware()

        self.collection.delete_many(
            {'timestamp': {'$lt': (now - timedelta(days=days_ago)).strftime("%Y-%m-%d %H:%M:%S")}}
        )

        self.collection.delete_many({'date_forward': {'$lt': now.strftime("%Y-%m-%d")}})

    def count(self):
        return self.collection.count()


min_price_storage = MinPriceStorage()
