# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from django.conf import settings
from pybreaker import CircuitBreaker
from pymongo import ReadPreference

from common.data_api.platforms.serialization import PlatformData, PlatformKey
from common.db.mongo import databases
from common.db.mongo.bulk_buffer import BulkBuffer
from common.dynamic_settings.core import DynamicSetting
from common.dynamic_settings.default import conf
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.library.python.tracing.instrumentation import traced_function


log = logging.getLogger(__name__)


define_setting(
    'DYNAMIC_PLATFORMS_DB_ALIAS',
    {Configuration.PRODUCTION: 'platforms', Configuration.STRESS: 'platforms'},
    default='default'
)
define_setting('DYNAMIC_PLATFORMS_BREAKER_PARAMS', default={'fail_max': 2, 'reset_timeout': 120})

conf.register_settings(
    DYNAMIC_PLATFORMS_ENABLED=DynamicSetting(True, cache_time=60, description='Включить динамические платформы')
)


platform_breaker = CircuitBreaker(**settings.DYNAMIC_PLATFORMS_BREAKER_PARAMS)


def get_dynamic_platform_collection():
    return databases[
        settings.DYNAMIC_PLATFORMS_DB_ALIAS
    ].dynamic_platforms.with_options(read_preference=ReadPreference.SECONDARY_PREFERRED)


class PlatformsClient(object):
    """
    Клиент для работы с динамическими платформами
    """

    def __init__(self, collection=None):
        self._coll = collection

    @property
    def coll(self):
        if not self._coll:
            self._coll = get_dynamic_platform_collection()
        return self._coll

    @platform_breaker
    @traced_function
    def find_platforms(self, keys):
        """
        :param keys: list of PlatformKey
        :return: dict of platforms by PlatformKey
        """
        if not conf.DYNAMIC_PLATFORMS_ENABLED:
            return {}

        or_criteria = [
            {'date': f.date.isoformat(), 'station_id': f.station_id, 'train_number': f.train_number} for f in keys
        ]
        docs = self.coll.find({'$or': or_criteria})
        result = {PlatformKey.create(d): PlatformData.create(d) for d in docs}
        return result

    @platform_breaker
    @traced_function
    def update(self, records):
        with BulkBuffer(self.coll, max_buffer_size=5000) as buff:
            for rec in records:
                update_set = {}
                if rec.data.departure_platform:
                    update_set['departure_platform'] = rec.data.departure_platform
                if rec.data.arrival_platform:
                    update_set['arrival_platform'] = rec.data.arrival_platform
                buff.update_one(rec.key.to_mongo_dict(), {'$set': update_set}, upsert=True)
