# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from common.data_api.platforms.serialization import PlatformKey


log = logging.getLogger(__name__)


class SegmentPlatformsBatch(object):
    @staticmethod
    def get_departure_safe(instance, segment, default):
        if instance:
            return instance.get_departure(segment, default)
        else:
            return default

    @staticmethod
    def get_arrival_safe(instance, segment, default):
        if instance:
            return instance.get_arrival(segment, default)
        else:
            return default

    @staticmethod
    def _get_simple_numbers(segment):
        return segment.thread.number.split('/')

    def __init__(self):
        self.platforms = []

    def try_load(self, client, segments):
        """
        :param segments: list of RThreadSegment
        """
        self.platforms = []
        if not segments:
            return True

        try:
            platform_keys = []
            for seg in segments:
                for simple_number in SegmentPlatformsBatch._get_simple_numbers(seg):
                    platform_keys.append(PlatformKey(
                        date=seg.departure.date(), station_id=seg.station_from.id, train_number=simple_number
                    ))
                    platform_keys.append(PlatformKey(
                        date=seg.arrival.date(), station_id=seg.station_to.id, train_number=simple_number
                    ))
            if platform_keys:
                self.platforms = client.find_platforms(platform_keys)
        except Exception as ex:
            log.exception('Ошибка при запросе динамических платформ для сегментов: {}'.format(ex.message))
            return False

        return True

    def _get_event(self, segment, segment_event_dt_attr, segment_station_attr, platform_attr, default_value):
        if not self.platforms:
            return default_value

        try:
            for simple_number in SegmentPlatformsBatch._get_simple_numbers(segment):
                platform = self.platforms.get(PlatformKey(
                    date=getattr(segment, segment_event_dt_attr).date(),
                    station_id=getattr(segment, segment_station_attr).id,
                    train_number=simple_number
                ))
                if platform:
                    # Only one departure/arrival this day from the station
                    return getattr(platform, platform_attr, default_value)
        except Exception as ex:
            log.exception('Ошибка при возврате динамической платформы сегтмента для события: {}'.format(ex.message))

        return default_value

    def get_departure(self, segment, default_value=None):
        return self._get_event(segment, 'departure', 'station_from', 'departure_platform', default_value)

    def get_arrival(self, segment, default_value=None):
        return self._get_event(segment, 'arrival', 'station_to', 'arrival_platform', default_value)


class PathPlatformsBatch(object):
    def _get_arrival_date(self, rtstation):
        return rtstation.get_arrival_dt(self.naive_start_dt, rtstation.station.pytz).date()

    def _get_departure_date(self, rtstation):
        return rtstation.get_departure_dt(self.naive_start_dt, rtstation.station.pytz).date()

    def __init__(self, naive_start_dt, train_number):
        self.naive_start_dt = naive_start_dt
        self.simple_numbers = []
        if train_number:
            self.simple_numbers = train_number.split('/')
        self.last_rtstation = None
        self.platforms = []

    def try_load(self, client, path):
        """
        :param path: list of rtstations in the right order
        """
        self.platforms = []
        if not path or not self.simple_numbers:
            return

        self.last_rtstation = path[-1]

        try:
            platform_keys = []
            for rts in path[:-1]:
                for simple_number in self.simple_numbers:
                    platform_keys.append(PlatformKey(
                        date=self._get_departure_date(rts), station_id=rts.station_id, train_number=simple_number
                    ))
            for simple_number in self.simple_numbers:
                platform_keys.append(PlatformKey(
                    date=self._get_arrival_date(self.last_rtstation),
                    station_id=self.last_rtstation.station_id,
                    train_number=simple_number
                ))
            if platform_keys:
                self.platforms = client.find_platforms(platform_keys)
        except Exception as ex:
            log.exception('Ошибка при запросе динамических платформ для пути: {}'.format(ex.message))

    def get_platform(self, rtstation, default_value=None):
        if not self.platforms:
            return default_value

        try:
            for simple_number in self.simple_numbers:
                if rtstation == self.last_rtstation:
                    platform = self.platforms.get(PlatformKey(
                        date=self._get_arrival_date(rtstation),
                        station_id=rtstation.station_id, train_number=simple_number
                    ))
                    if platform:
                        return platform.arrival_platform  # only one departure this day from the station
                else:
                    platform = self.platforms.get(PlatformKey(
                        date=self._get_departure_date(rtstation),
                        station_id=rtstation.station_id, train_number=simple_number
                    ))
                    if platform:
                        return platform.departure_platform  # only one departure this day from the station
        except Exception as ex:
            log.exception('Ошибка при возврате динамической платформы станции: {}'.format(ex.message))
        return default_value


class ScheduleRoutePlatformsBatch(object):
    @staticmethod
    def get_departure_safe(instance, route, default):
        if instance:
            return instance.get_departure(route, default)
        else:
            return default

    @staticmethod
    def get_arrival_safe(instance, route, default):
        if instance:
            return instance.get_arrival(route, default)
        else:
            return default

    @staticmethod
    def _get_simple_numbers(route):
        return route.thread.number.split('/')

    def __init__(self):
        self.platforms = []

    def try_load(self, client, routes):
        """
        :param routes: list of ScheduleRoute
        """
        self.platforms = []
        if not routes:
            return

        try:
            platform_keys = []
            for route in routes:
                for simple_number in ScheduleRoutePlatformsBatch._get_simple_numbers(route):
                    if route.departure_dt:
                        platform_keys.append(PlatformKey(
                            date=route.departure_dt.date(),
                            station_id=route.rtstation.station_id,
                            train_number=simple_number
                        ))
                    if route.arrival_dt and (
                        not route.departure_dt or route.departure_dt.date() != route.arrival_dt.date()
                    ):  # if key is different
                        platform_keys.append(PlatformKey(
                            date=route.arrival_dt.date(),
                            station_id=route.rtstation.station_id,
                            train_number=simple_number
                        ))
            if platform_keys:
                self.platforms = client.find_platforms(platform_keys)
        except Exception as ex:
            log.exception('Ошибка при запросе динамических платформ для расписания: {}'.format(ex.message))

    def _get_event(self, route, route_event_dt_attr, platform_attr, default_value):
        if not self.platforms:
            return default_value

        event_dt = getattr(route, route_event_dt_attr, None)
        if not event_dt:
            return default_value

        try:
            for simple_number in ScheduleRoutePlatformsBatch._get_simple_numbers(route):
                platform = self.platforms.get(PlatformKey(
                    date=event_dt.date(), station_id=route.rtstation.station_id, train_number=simple_number
                ))
                if platform:
                    # Only one departure/arrival this day from the station
                    return getattr(platform, platform_attr, default_value)
        except Exception as ex:
            log.exception('Ошибка при возврате динамической платформы маршрута для события: {}'.format(ex.message))

        return default_value

    def get_departure(self, route, default_value=None):
        return self._get_event(route, 'departure_dt', 'departure_platform', default_value)

    def get_arrival(self, route, default_value=None):
        return self._get_event(route, 'arrival_dt', 'arrival_platform', default_value)
