# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import itertools
import re
from collections import defaultdict

from common.data_api.base import BaseOAuthApi


class QloudPublicApi(BaseOAuthApi):
    PERSISTENT_DNS_TEMPLATE = '{instance}.{component}.{environment}.{application}.{project}.stable.qloud-d.yandex.net'
    HOST_DC_RE = re.compile(r'^[A-Za-z]+')

    def __init__(self, token, base_url='https://platform.yandex-team.ru/api/v1/', **kwargs):
        super(QloudPublicApi, self).__init__(token, base_url, **kwargs)

    def dump_environment(self, environment_id):
        return self._process_json_query('get', 'environment/dump/{}'.format(environment_id))

    def upload_environment(self, environment_dump):
        return self._process_json_query('post', 'environment/upload', json=environment_dump)

    def get_environment_info(self, environment_id):
        return self._process_json_query('get', 'environment/stable/{}'.format(environment_id))

    @staticmethod
    def _dump_component(env_dump, component_name):
        for component in env_dump['components']:
            if component['componentName'] == component_name:
                return component

    @classmethod
    def get_dc_from_host(cls, host):
        return cls.HOST_DC_RE.search(host).group()

    def get_component_environment_variables(self, environment_id, component_name):
        env_dump = self.dump_environment(environment_id)
        return self._dump_component(env_dump, component_name)['environmentVariables']

    def update_component_environment_variables(self, environment_id, component_name, **kwargs):
        env_dump = self.dump_environment(environment_id)
        component_dump = self._dump_component(env_dump, component_name)
        component_dump['environmentVariables'].update(kwargs)
        self.upload_environment(env_dump)

    def get_dc_grouped_hosts(self, component_id, use_persistent_names=False):
        environment_id, component_name = component_id.rsplit('.', 1)
        dump = self.get_environment_info(environment_id)
        result = defaultdict(list)
        for current_component_name, component in dump.get('components', {}).items():
            if current_component_name == component_name or component_name == '*':
                for instance in component.get('runningInstances', []):
                    host = instance.get('host')
                    if not host:
                        continue
                    if use_persistent_names:
                        host_persistent_name = self.PERSISTENT_DNS_TEMPLATE.format(
                            instance=instance['instanceId'],
                            component=current_component_name,
                            environment=component['environmentName'],
                            application=component['applicationName'],
                            project=component['projectName'],
                        )
                        result[self.get_dc_from_host(host)].append(host_persistent_name)
                    else:
                        result[self.get_dc_from_host(host)].append(host)
        return result

    def get_hosts(self, component_id, use_persistent_names=False):
        return list(itertools.chain(*list(self.get_dc_grouped_hosts(component_id, use_persistent_names).values())))
