# coding: utf8
from itertools import islice

from common.db.mongo import database


class SearchStats(object):
    def __init__(self, collection_from=None, collection_to=None):
        self._collection_from = collection_from
        self._collection_to = collection_to

    @property
    def collection_from(self):
        if not self._collection_from:
            self._collection_from = database.search_stats_from
        return self._collection_from

    @property
    def collection_to(self):
        if not self._collection_to:
            self._collection_to = database.search_stats_to
        return self._collection_to


    def get_top(self, collection, obj_key, t_type, search_type, limit, geo_id):
        """
        Search in a collection with documents of this scheme (see SearchStatsSaver):
        {
            't_type': 'all',
            'obj_type': 'c',
            'obj_id': '213',
            'search_type': 'c',  # one of ['c', 's', 'all'], to search only cities, stations or both
            'top_searches': [
                {
                    'geo_id': 'all',  # stats sum for all geo_ids
                    'data': [
                        {'obj_type': 42, 'obj_id': 111},
                        {'obj_type': 43, 'obj_id': 12}
                    ]
                },
                {
                    'geo_id': 213,
                    'data': [
                        {'obj_type': 'c', 'obj_id': 42, 'total': 111},
                        {'obj_type': 's', 'obj_id': 43, 'total': 12}
                    ]
                }
            ]
        }
        """
        obj_type, obj_id = obj_key[0], obj_key[1:]

        query = {
            't_type': t_type,
            'obj_type': obj_type,
            'obj_id': int(obj_id),
            'search_type': search_type,
            'top_searches.geo_id': geo_id,
        }
        project = {'top_searches': {'$elemMatch': {'geo_id': geo_id}}}

        doc = collection.find_one(query, project)
        if not doc:
            return

        # top_searches has only one element because of previous $elemMatch by geo_id
        doc_for_geo_id = doc['top_searches'][0]
        for item in islice(doc_for_geo_id['data'], limit):
            key = item['obj_type'] + str(item['obj_id'])
            yield key, item['total']

    def get_top_from(self, obj_key, t_type, search_type, geo_id='all', limit=20):
        return self.get_top(self.collection_from, obj_key, t_type, search_type, limit=limit, geo_id=geo_id)

    def get_top_to(self, obj_key, t_type, search_type, geo_id='all', limit=20):
        return self.get_top(self.collection_to, obj_key, t_type, search_type, limit=limit, geo_id=geo_id)


search_stats = SearchStats()
