from collections import defaultdict
from operator import itemgetter

from common.db.mongo.bulk_buffer import BulkBuffer
from travel.rasp.library.python.common23.date import environment


class SearchStatsSaver(object):
    def __init__(self, collection_from, collection_to, top_size=50, logger=None):
        self.collection_from = collection_from
        self.collection_to = collection_to
        self.top_size = top_size
        self.logger = logger

        index = [("t_type", 1), ("obj_type", 1), ("obj_id", 1), ("search_type", 1), ("top_searches.geo_id", 1)]
        self.collection_from.create_index(index, background=True)
        self.collection_to.create_index(index, background=True)

    def save_stats(self, stats):
        """
        :stats:
        {
            'suburban': {  # transport type
                ('c', 213): {  # obj_from
                    ('c', 2): {  # obj_to
                        54: 100500,  # geo_id: total searches
                        ...
                    },
                    ...
                },
                ...
            },
            ...
        }

        Save stats to self.collection_from and self.collection_to in this way:
        {
            't_type': 'suburban',  # one of transport types or 'all'
            'obj_type': 'c',  # c or s
            'obj_id': 213,
            'search_type': 'c',  // one of ['c', 's', 'all'], to search only cities, stations or both
            'top_searches': [  # top searches for each geo_id
                {
                    'geo_id': 'all',  # stats sum for all geo_ids
                    'data': [
                        {'obj_type': 42, 'obj_id': 111},
                        {'obj_type': 43, 'obj_id': 12}
                    ]
                },
                {
                    'geo_id': 213,
                    'data': [
                        {'obj_type': 'c', 'obj_id': 42, 'total': 111},
                        {'obj_type': 's', 'obj_id': 43, 'total': 12}
                    ]
                }
            ]
        }
        """

        def add_item(data, search_type, obj_type, obj_id, geo_id, total):
            data[search_type][geo_id].append({
                'obj_type': obj_type,
                'obj_id': obj_id,
                'total': total,
            })

        def save(collection, obj_type, obj_id, data):
            for search_type, by_geo_id in data.items():
                top_searches = []
                for geo_id, data in by_geo_id.items():
                    data = sorted(data, key=itemgetter('total'), reverse=True)[:self.top_size]
                    top_searches.append({
                        'geo_id': geo_id,
                        'data': data,
                    })

                collection.update(
                    {
                        't_type': t_type,
                        'obj_type': obj_type,
                        'obj_id': obj_id,
                        'search_type': search_type,
                    },
                    {
                        '$set': {
                            'top_searches': top_searches,
                            'update_time': begin_dt
                        }
                    },
                    upsert=True,
                )

        coll_from = BulkBuffer(self.collection_from, logger=self.logger)
        coll_to = BulkBuffer(self.collection_to, logger=self.logger)
        begin_dt = environment.now()

        with coll_from, coll_to:
            for t_type, by_obj_from in stats.items():

                to_data_by_obj = defaultdict(lambda: defaultdict(lambda: defaultdict(list)))
                for (obj_from_type, obj_from_id), by_obj_to in by_obj_from.items():

                    from_data = defaultdict(lambda: defaultdict(list))
                    for (obj_to_type, obj_to_id), by_geo_id in by_obj_to.items():

                        to_data = to_data_by_obj[(obj_to_type, obj_to_id)]
                        total_for_geoid = 0
                        for geo_id, total in by_geo_id.items():
                            add_item(from_data, obj_to_type, obj_to_type, obj_to_id, geo_id, total)
                            add_item(from_data, 'all', obj_to_type, obj_to_id, geo_id, total)

                            add_item(to_data, obj_from_type, obj_from_type, obj_from_id, geo_id, total)
                            add_item(to_data, 'all', obj_from_type, obj_from_id, geo_id, total)

                            total_for_geoid += total

                        add_item(from_data, obj_to_type, obj_to_type, obj_to_id, 'all', total_for_geoid)
                        add_item(from_data, 'all', obj_to_type, obj_to_id, 'all', total_for_geoid)

                        add_item(to_data, 'all', obj_from_type, obj_from_id, 'all', total_for_geoid)
                        add_item(to_data, obj_from_type, obj_from_type, obj_from_id, 'all', total_for_geoid)

                    save(coll_from, obj_from_type, obj_from_id, from_data)

                for (obj_to_type, obj_to_id), to_data in to_data_by_obj.items():
                    save(coll_to, obj_to_type, obj_to_id, to_data)

        self.collection_from.delete_many({'$or': [
            {'update_time': {'$exists': False}},
            {'update_time': {'$lt': begin_dt}}
        ]})
        self.collection_to.delete_many({'$or': [
            {'update_time': {'$exists': False}},
            {'update_time': {'$lt': begin_dt}}
        ]})
