# coding: utf-8
from __future__ import unicode_literals

import codecs
import json
import logging
from collections import namedtuple

import requests
from django.conf import settings
from django.utils.encoding import force_text, force_bytes

from common.settings.configuration import Configuration
from common.settings.utils import define_setting

log = logging.getLogger(__name__)


define_setting('YASENDR_HOST', {
    Configuration.PRODUCTION: 'sender.yandex-team.ru',
    Configuration.TESTING: 'test.sender.yandex-team.ru',
    Configuration.DEVELOPMENT: 'test.sender.yandex-team.ru'
}, default=None)

SSL_VERIFY = False  # TODO убрать после переезда на trusty

define_setting('YASENDR_ACCOUNT', default='rasp')
define_setting('YASENDR_AUTH_TOKEN', {
    Configuration.PRODUCTION: '9a900569c6f14930a311894f1668bf18',
    Configuration.TESTING: 'dd0ac40d0fcf431d9f9d7c251de7ecd3',
    Configuration.DEVELOPMENT: 'dd0ac40d0fcf431d9f9d7c251de7ecd3'
}, default=None)

define_setting('YASENDR_FOR_TESTING', default=False)

define_setting('YASENDR_TIMEOUT', default=20)


class Error(Exception):
    pass


class ParsingError(Error):
    pass


class ErrorResponse(Error):
    pass


class Attachment(namedtuple('Attachment', ('filename', 'mime_type', 'content'))):
    def dump(self):
        return {
            'filename': self.filename,
            'mime_type': self.mime_type,
            'data': codecs.encode(force_bytes(self.content), 'base64')
        }


class Campaign(namedtuple('Campaign', ('account', 'campaign', 'auth_token'))):
    def send(self, to_email, args, attachments=()):
        url = 'https://{HOST}/api/0/{self.account}/transactional/{self.campaign}/send'.format(
            HOST=settings.YASENDR_HOST, self=self)
        data = {'args': json.dumps(args)}
        if attachments:
            data['attachments'] = json.dumps([attachment.dump() for attachment in attachments])

        params = {'to_email': to_email}
        if settings.YASENDR_FOR_TESTING:
            # чтобы при разработке можно было отправлять письма из черновиков
            # после активации письмо также будет отправляться
            # в этом режиме можно отправлять письмо только владельцу рассылки
            params['for_testing'] = True

        log.debug('Request: %s, params: %s, args: %s', url, force_text(params), force_text(args))
        response = requests.post(url, params=params, data=data,
                                 auth=(self.auth_token, ''), timeout=settings.YASENDR_TIMEOUT, verify=SSL_VERIFY)
        log.debug('Result: %s', force_text(response.text))
        self._check_result(response)

    def _check_result(self, response):
        response_data = {}
        try:
            response.raise_for_status()
        except Exception:
            raise ErrorResponse('Email sending error. http_code: {} text: {}'.format(response.status_code,
                                                                                     response.text))

        try:
            response_data = response.json()
        except Exception:
            raise ParsingError('Sendr response is not json: {}'.format(response.text))

        try:
            result = response_data['result']
        except KeyError:
            raise ParsingError('Sendr response doesn\'t have result field')

        try:
            status = result['status']
        except KeyError:
            raise ParsingError('Sendr result doesn\'t have status field')

        if status == 'ERROR':
            # в документации написано, что текст ошибки приходит в message, по факту - в error
            raise ErrorResponse('Email sending error: {} message: {}'.format(result.get('error'),
                                                                             result.get('message')))

        if status != 'OK':
            raise ParsingError('Sendr response has invalid status value {!r}'.format(status))

    @classmethod
    def create_rasp_campaign(cls, campaign_code):
        return cls(
            account=settings.YASENDR_ACCOUNT,
            campaign=campaign_code,
            auth_token=settings.YASENDR_AUTH_TOKEN
        )
