# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
import re

import requests
import six
from django.conf import settings

from common.data_api.sup.config import *  # noqa
from common.dynamic_settings.core import DynamicSetting
from common.dynamic_settings.default import conf

conf.register_settings(
    SUP_PUSH_TTL=DynamicSetting(
        3000,
        cache_time=10 * 60,
        description='Устанавливает ttl в секундах для пушей, отправляемых через API СУПа'
    )
)

log = logging.getLogger(__name__)


class SupClient(object):
    """
    Клиент АПИ СУПа - сервиса для работы с пушами.

    https://abc.yandex-team.ru/services/sup/
    https://push-beta.n.yandex-team.ru/docs/api-guide.html
    https://push-beta.n.yandex-team.ru/docs/api-guide-v2.html
    """

    @six.python_2_unicode_compatible
    class InvalidRequest(Exception):
        def __init__(self, msg, errors):
            self.msg = msg
            self.errors = errors

        def __repr__(self):
            return 'InvalidRequest({}: {})'.format(self.msg, '; '.join(self.errors)).encode('utf8')

        def __str__(self):
            return self.__repr__()

    def __init__(self, sup_url=None, oauth_token=None):
        self.sup_url = sup_url
        self.oauth_token = oauth_token

    def call(self, method, data=None, params=None, path=''):
        http_method = 'post' if data else 'get'
        url = '{}/{}/{}'.format(self.sup_url, method, path)

        json_data = json.dumps(data)
        if http_method == 'post':
            log.info('Calling {} {}: {}'.format(url, params, json_data))

        response = requests.request(
            http_method,
            url,
            params=params,
            headers={
                'Content-Type': 'application/json;charset=UTF-8',
                'Authorization': 'OAuth {}'.format(self.oauth_token),
            },
            data=json_data,
            verify=False,
        )

        try:
            response.raise_for_status()
        except requests.HTTPError:
            log.exception(response.request.body)
            log.exception(response.text)
            raise

        return response.text

    def pushes(
            self, project, receivers, title, text, device_id_policy, install_id_policy=None,
            image=None, image_url=None, url=None, data=None,
            ttl=conf.SUP_PUSH_TTL, max_expected_receivers=None, push_id=None, dry_run=True, high_priority=False
        ):

        """
        :param project: проект в СУПе
        :param receivers: список получателей, формат см. в доке
        :param title: заголовк пуша
        :param text: текст пуша
        :param device_id_policy: политика пуша. Идентификатор лимита уведомлений при адресации по did
        :param install_id_policy: политика пуша. Идентификатор лимита уведомлений при адресации по uuid
        :param image: картинка пуша - строковой идентификатор ресурса, зашитого в приложение
        :param image_url: урл картинки пуша. Осторожно, все клиенты полезут за этой картинкой одновременно!
        :param url: урл при клике на пуш
        :param ttl: время жизни (устройства, подключившиеся позже отправки пуша, но пока он жив - получат этот пуш)
        :param max_expected_receivers: Ограничение количества получателей пуша. Если при отправке СУП поймет,
               что пуш уйдет на слишком большое количество устройств, то пуш не отправится.
        :param data: данные, передававемые в приложение при клике на пуш
        :param push_id: идентификатор пуша, виден в логах. На стороне СУПа к нему добавляется рандомный хвост
        :param dry_run: если True, то пуш не будет отправлен, но будет возвращена дебажная информация об отправке.
        :param high_priority: пуши будут обрабатываться отдельной очередью СУПа, не зависящей от массовых рассылок
        """
        request, params = SupClient.get_push_request(
            project, receivers, title, text, device_id_policy, install_id_policy, image, image_url, url, data, ttl,
            max_expected_receivers, push_id, dry_run, high_priority
        )

        try:
            return self.call('pushes', request, params=params)
        except requests.HTTPError as e:
            if e.response.status_code == 400:
                response = e.response.json()
                msgs = []
                for err in response.get('errors', []):
                    msgs.append(
                        '"{}" = "{}": {}'.format(err.get('field'), err.get('rejectedValue'), err.get('defaultMessage'))
                    )

                raise SupClient.InvalidRequest('SUP validation failed', errors=msgs)
            else:
                raise

    @classmethod
    def get_push_request(
            cls, project, receivers, title, text, device_id_policy, install_id_policy=None,
            image=None, image_url=None, url=None, data=None,
            ttl=conf.SUP_PUSH_TTL, max_expected_receivers=None, push_id=None, dry_run=True, high_priority=False
    ):
        data = data or {}
        data.update({
            "push_id": push_id if push_id else project,
            "push_uri": url,
        })

        # по умолчанию текст пуша и полный текст в приложении - одинаковы
        if 'text' not in data:
            data['text'] = text

        request = {
            "project": project,
            "max_expected_receivers": max_expected_receivers,
            "ttl": ttl,
            "schedule": "now",
            "adjust_time_zone": False,
            "receiver": receivers,
            "notification": {
                "title": title,
                "body": text,
                "iconId": image,
                "icon": image_url,
            },
            "ios_features": {
                "soundType": 0  # default sound
            },
            "throttle_policies": {
                "device_id": device_id_policy
            },
            "data": data,
        }

        if install_id_policy:
            request['throttle_policies']['install_id'] = install_id_policy

        params = {}
        if dry_run:
            params['dry_run'] = 1

        if high_priority:
            params['priority'] = 'high'

        return request, params

    def pushes_batch_yt(self, table):
        """Устаревший способ групповой отправки пушей"""
        return self.call(
            method='pushes',
            data={'path': table},
            path='batch'
        )

    @classmethod
    def get_receivers_number(cls, dry_run_response):
        """
        Пытаемся достать количество потенциальных получателей пуша из дебажного ответа.
        Нельзя расчитывать на 100% стабильность этого интерфейса, поэтому если не можем распарсить - забиваем.
        """
        try:
            for line in dry_run_response['trace']['resolveEvents']:
                match = re.match('.*resolved to ([0-9]+) receiver.*', line)
                if match:
                    return int(match.group(1))
        except Exception:
            log.exception("Can't parse dry-run format: {}".format(dry_run_response))

    @staticmethod
    def get_app_id_tag(app_ids):
        return "tag:app_id in ({})".format(', '.join("'{}'".format(app_id) for app_id in app_ids))


def get_client(sup_url=None, oauth_token=None):
    return SupClient(
        sup_url=sup_url or settings.SUP_URL,
        oauth_token=oauth_token or settings.SUP_OAUTH_TOKEN,
    )
