# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from multiprocessing import TimeoutError
from multiprocessing.dummy import Pool

from django.conf import settings
from six.moves.queue import Queue

from common.data_api.ticket_daemon.query import AsyncResult


log = logging.getLogger(__name__)


class TicketDaemonAsyncClient(object):
    """
    Пример:
        client = TicketDaemonAsyncClient(10)
        for query, variants, statuses, err in client.get_results(queries, True):
            pass
    """

    def __init__(self, pool_size):
        self.pool_size = pool_size

    @staticmethod
    def process_query(send_query, results_queue, query):
        query.execute(results_queue, send_query)

    def get_results(self, queries, send_query, timeout=0):
        """
        Выполняет запросы к тикет демону в ThreadPool.
        :param queries: список запросов к тикет демону (Query или QueryQidPoll).
        :param send_query: требуется ли инициализировать поиск цен.
        :param timeout: суммарный таймаут на запрос.
        :return: список кортежей вида (запрос, варианты, статусы).
        """

        queue = Queue()
        queries = list(queries)
        pool = Pool(processes=self.pool_size)
        try:
            pool.map_async(lambda q: q.execute(queue, send_query), queries).get(timeout)
        except TimeoutError:
            log.exception(u'Timeout при попытке получить цены от ticket-daemon')
        finally:
            pool.terminate()

        results = [queue.get() for _ in range(queue.qsize())]
        processed_queries = set(str(r.query) for r in results)
        all_queries = {str(q): q for q in queries}
        for key, query in all_queries.items():
            if key not in processed_queries:
                results.append(AsyncResult(query, [], [], TimeoutError))
        return results


def get_results_async(queries, send_query=False, timeout=settings.TICKET_ASYNC_CLIENT_TIMEOUT):
    valid_queries = [query for query in queries if query.is_valid()]
    return (TicketDaemonAsyncClient(settings.TICKET_DAEMON_ASYNC_CLIENT_POOL_SIZE)
            .get_results(valid_queries, send_query, timeout))
