# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from abc import ABCMeta, abstractmethod
from collections import defaultdict
from multiprocessing import TimeoutError

import six
from requests.exceptions import Timeout

from common.data_api.ticket_daemon.instance import ticket_daemon_client
from common.data_api.ticket_daemon.serialization.query_search_params import QuerySearchParamsSchema
from common.data_api.ticket_daemon.serialization.query_qid_poll_params import QueryQidPollParamsSchema
from common.data_api.ticket_daemon.serialization.variant import parse_variants
from common.models.transport import TransportType
from common.utils.namedtuple import namedtuple_with_defaults


log = logging.getLogger(__name__)


class AsyncResult(namedtuple_with_defaults('AsyncResult',
                                           ['query', 'variants', 'statuses', 'error'],
                                           defaults={'error': None})):
    @property
    def querying(self):
        querying = False
        if self.statuses and 'querying' in self.statuses.values():
            querying = True
        elif self.error:
            querying = isinstance(self.error, (Timeout, TimeoutError))
        return querying


@six.add_metaclass(ABCMeta)
class AsyncDaemonQuery(object):
    QUERY_SCHEMA = None
    """Интерфейс для асинхронного выполнения запросов в async_api"""

    def _make_search_params(self):
        query_args, _errors = self.QUERY_SCHEMA(strict=True).dump({
            field: value
            for field, value in self._asdict().items()
            if value is not None
        })
        return query_args

    @abstractmethod
    def is_valid(self):
        """Проверка запроса"""

    @abstractmethod
    def collect_variants(self):
        """Получение результатов запроса"""

    @abstractmethod
    def execute(self, results_queue, send_init_query):
        """Выполнение запроса и публикация результата в result_queue"""


class Query(AsyncDaemonQuery, namedtuple_with_defaults('Query', (
    'national_version',
    'service',
    'point_from',
    'point_to',
    'date_forward',
    'date_backward',
    'klass',
    'passengers',
    't_code',
    'yandexuid',

    # не передаются
    'partners',
    'user_settlement',
))):
    QUERY_SCHEMA = QuerySearchParamsSchema

    def _check_t_code(self):
        if self.t_code != 'plane':
            raise ValueError('Ticket Daemon is for planes only')

    def query_all(self, ignore_cache=False):
        self._check_t_code()
        result = ticket_daemon_client.init_search(dict(self._make_search_params(), ignore_cache=ignore_cache))
        return result['qid']

    def collect_variants(self):
        self._check_t_code()
        result = ticket_daemon_client.get_results_by_search_params(self._make_search_params())
        if not result:
            return defaultdict(list), []

        return parse_variants(result['variants'], result['reference']), result['status']

    def is_valid(self):
        """Проверяем что через point_from и point_to ходит транспорт с указанным t_code.
        На данный момент проверка выполняется только для самолетов."""
        t_type = TransportType.objects.get(code=self.t_code)
        if t_type.id == TransportType.PLANE_ID:
            return all(self._is_plane_point(point) for point in (self.point_from, self.point_to))
        return True

    def execute(self, results_queue, send_init_query):
        if send_init_query:
            try:
                self.query_all()
            except Exception as ex:
                log.exception(
                    u'Ошибка при попытке сделать запрос point_from=%(point_from)s point_to=%(point_to)s'
                    u' date_forward=%(date_forward)s national_version=%(national_version)s t_code=%(t_code)s'
                    u' yandexuid=%(yandexuid)s',
                    {
                        'point_from': self.point_from.point_key,
                        'point_to': self.point_to.point_key,
                        'date_forward': self.date_forward,
                        'national_version': self.national_version,
                        't_code': self.t_code,
                        'yandexuid': self.yandexuid,
                    }
                )
                results_queue.put(AsyncResult(self, [], [], ex))
                return

        try:
            variants, statuses = self.collect_variants()
            results_queue.put(AsyncResult(self, variants, statuses))
        except Exception as ex:
            log.exception(
                u'Ошибка при попытке получить цены point_from=%(point_from)s point_to=%(point_to)s'
                u' date_forward=%(date_forward)s national_version=%(national_version)s t_code=%(t_code)s'
                u' yandexuid=%(yandexuid)s',
                {
                    'point_from': self.point_from.point_key,
                    'point_to': self.point_to.point_key,
                    'date_forward': self.date_forward,
                    'national_version': self.national_version,
                    't_code': self.t_code,
                    'yandexuid': self.yandexuid,
                }
            )
            results_queue.put(AsyncResult(self, [], [], ex))

    @staticmethod
    def _is_plane_point(point):
        return bool(point.iata_point or point.sirena_point)


class QueryQidPoll(AsyncDaemonQuery, namedtuple_with_defaults('QueryQidPoll', (
    'qid',
    'skip_partners',
    'yandexuid',
))):
    QUERY_SCHEMA = QueryQidPollParamsSchema

    def collect_variants(self):
        result = ticket_daemon_client.get_results_by_qid(self._make_search_params())
        if not result:
            return defaultdict(list), []

        return parse_variants(result['variants'], result['reference']), result['status']

    def is_valid(self):
        return True

    def execute(self, results_queue, send_init_query=False):
        try:
            variants, statuses = self.collect_variants()
            results_queue.put(AsyncResult(self, variants, statuses))
        except Exception as ex:
            log.exception(
                u'Ошибка при попытке получить цены по qid=%(qid)s yandexuid=%(yandexuid)d',
                {
                    'qid': self.qid,
                    'yandexuid': self.yandexuid,
                }
            )
            results_queue.put(AsyncResult(self, [], [], ex))
