# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.utils.functional import cached_property
from marshmallow import Schema, fields, post_load

from common.models.schedule import Company
from common.models.transport import TransportType
from common.serialization.fields import DatetimeAwareField
from common.serialization.schema import get_defaults_from_schema
from common.utils.date import MSK_TZ
from common.utils.namedtuple import namedtuple_with_defaults
from common.utils.title_generator import DefaultTitleGenerationStrategy, TitleGenerator


class SegmentSchema(Schema):
    # Flight
    display_t_code = fields.String(load_from='t_type_code')
    supplier_code = fields.String(allow_none=True)
    number = fields.String()
    station_from = fields.Method(deserialize='station_by_pk')
    station_to = fields.Method(deserialize='station_by_pk')
    departure = DatetimeAwareField()
    arrival = DatetimeAwareField()
    company = fields.Method(deserialize='company_by_pk', allow_none=True)
    forward_key = fields.String()  # https://st.yandex-team.ru/RASPFRONT-2711
    url = fields.String()

    # RaspRoute
    electronic_ticket = fields.Boolean(allow_none=True)

    # UkrmintransTrainRoute, UkrmintransBusRoute
    first_station = fields.Method(deserialize='station_by_pk', allow_none=True)
    last_station = fields.Method(deserialize='station_by_pk', allow_none=True)
    # show_number = fields.Boolean()

    # BlablacarRoute
    number_of_variants = fields.Integer(allow_none=True)
    duration_in_seconds = fields.Integer(allow_none=True)

    @post_load
    def post_load(self, data):
        return Segment(**data)

    def station_by_pk(self, pk):
        if not self.context:
            return None
        return self.context['stations'][pk]

    def company_by_pk(self, pk):
        return Company.objects.get(pk=pk)


class Segment(namedtuple_with_defaults('Segment',
                                       SegmentSchema().fields.keys(),
                                       get_defaults_from_schema(SegmentSchema))):
    gone = False
    rtstation_from = None
    rtstation_to = None
    thread = None

    @cached_property
    def t_type(self):
        return TransportType.objects.get(code=self.display_t_code)

    def get_popular_title(self, lang=None):
        return self.L_title(popular=True, lang=lang)

    @cached_property
    def title(self):
        return self.L_title()

    @cached_property
    def msk_departure(self):
        return self.departure and self.departure.astimezone(MSK_TZ).replace(tzinfo=None)

    @cached_property
    def msk_arrival(self):
        return self.arrival and self.arrival.astimezone(MSK_TZ).replace(tzinfo=None)

    @cached_property
    def duration(self):
        return self.arrival and self.departure and self.arrival - self.departure

    def L_title(self, **kwargs):
        generation_strategy = DefaultTitleGenerationStrategy()
        title_dict = generation_strategy.build_title_dict(None, (self.station_from, self.station_to))
        return TitleGenerator.L_title(
            title_dict, prefetched_points=generation_strategy.prefetched_points, **kwargs
        )
