# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import re
from datetime import timedelta

from django.conf import settings

from common.settings.configuration import Configuration
from common.settings.utils import define_setting


define_setting('ALIGN_SEARCH_SEGMENT_KEYS', {Configuration.PRODUCTION: True}, default=False)

TIME_RANGE_HOURS = 2
NUMBER_FORMAT = '{:03}{}'
TRAIN_KEY_FORMAT = 'train {} {}'

train_number_re = re.compile(r'(\d+)(\D*)$')


def _make_datetime_keys(departure, numbers):
    if settings.ALIGN_SEARCH_SEGMENT_KEYS:
        dt_keys = {
            make_datetime_key(departure - timedelta(hours=TIME_RANGE_HOURS / 2)),
            make_datetime_key(departure + timedelta(hours=TIME_RANGE_HOURS / 2))
        }
    else:
        dt_keys = {
            make_precise_datetime_key(departure)
        }

    return [
        TRAIN_KEY_FORMAT.format(number, dt_key)
        for number in numbers
        for dt_key in sorted(dt_keys)
    ]


def _make_segment_datetime_keys(segment, numbers):
    return _make_datetime_keys(segment.departure, numbers)


def get_reverse_train_number(train_digits):
    train_digits = int(train_digits)
    if train_digits % 2 == 0:
        train_digits -= 1
    else:
        train_digits += 1
    return train_digits


def get_possible_numbers(train_number):
    """
    Поезд может менять номер с четного на нечетный в зависимости от участка пути.
    Это необходимо учитывать при привязке цен, поэтому генерируем все возможные номера.
    """
    try:
        number, letter = _get_number_and_first_letter(train_number)
    except _StrangeTrainNumber:
        return [train_number]
    else:
        return [NUMBER_FORMAT.format(number, letter), NUMBER_FORMAT.format(get_reverse_train_number(number), letter)]


def make_datetime_key(dt):
    aligned_dt = dt.replace(hour=int(dt.hour // TIME_RANGE_HOURS) * TIME_RANGE_HOURS)
    return aligned_dt.strftime('%Y%m%d_%H')


def make_precise_datetime_key(dt):
    return dt.strftime('%Y%m%d_%H%M')


class _StrangeTrainNumber(Exception):
    pass


def _get_number_and_first_letter(train_number):
    match = train_number_re.match(train_number)
    if not match:
        raise _StrangeTrainNumber

    number, letters = match.groups()
    letter = letters[0] if letters else ''

    return int(number), letter
