# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging

import requests


log = logging.getLogger(__name__)


class DocumentTypes(object):
    RU_NATIONAL_PASSPORT = 'ru_national_passport'


class TravelersClient(object):
    """
    Client for Ya.Travel Traveler's notebook
    https://github.yandex-team.ru/avia/travelers/
    """
    def __init__(self, host, tvm_travelers_id, tvm_factory, timeout=None):
        self.host = host
        self.tvm_travelers_id = tvm_travelers_id
        self.tvm_factory = tvm_factory
        self.timeout = timeout

    def call(self, http_method, method, user_ticket=None, params=None, data=None):
        url = '{}/{}'.format(self.host, method)

        travelers_ticket = self.tvm_factory.get_provider().get_ticket(self.tvm_travelers_id)
        headers = {
            'X-Ya-Service-Ticket': travelers_ticket,
            'Content-Type': 'application/json',
        }
        if user_ticket:
            headers['X-Ya-User-Ticket'] = user_ticket

        log.info('Calling %s %s', method, url)
        response = requests.request(
            http_method,
            url,
            params=params,
            headers=headers,
            data=json.dumps(data),
            timeout=self.timeout
        )

        try:
            response.raise_for_status()
        except requests.HTTPError:
            log.exception(response.request.body)
            log.exception(response.text)
            raise

        return response.json()

    def create_traveler(self, uid, user_ticket, email, phone=None):
        data = {'email': email, 'agree': True}
        if phone:
            data['phone'] = phone

        return self.call('POST', 'v1/travelers/{}'.format(uid), user_ticket, data=data)

    def get_traveler(self, uid, user_ticket):
        try:
            return self.call('GET', 'v1/travelers/{}'.format(uid), user_ticket)
        except requests.HTTPError as ex:
            if ex.response.status_code == 404:
                return None
            else:
                raise

    def create_passenger(self, uid, user_ticket, data):
        return self.call('POST', 'v1/travelers/{}/passengers'.format(uid), user_ticket, data=data)

    def edit_passenger(self, uid, user_ticket, passenger_id, data):
        return self.call('PUT', 'v1/travelers/{}/passengers/{}'.format(uid, passenger_id), user_ticket, data=data)

    def get_passenger(self, uid, user_ticket, passenger_id):
        return self.call('GET', 'v1/travelers/{}/passengers/{}'.format(uid, passenger_id), user_ticket)

    def get_passengers(self, uid, user_ticket):
        return self.call('GET', 'v1/travelers/{}/passengers'.format(uid), user_ticket)

    def create_document(self, uid, user_ticket, passenger_id, data):
        url = 'v1/travelers/{}/passengers/{}/documents'.format(uid, passenger_id)
        return self.call('POST', url, user_ticket, data=data)

    def edit_document(self, uid, user_ticket, passenger_id, doc_id, data):
        url = 'v1/travelers/{}/passengers/{}/documents/{}'.format(uid, passenger_id, doc_id)
        return self.call('PUT', url, user_ticket, data=data)

    def get_document(self, uid, user_ticket, passenger_id, doc_id):
        url = 'v1/travelers/{}/passengers/{}/documents/{}'.format(uid, passenger_id, doc_id)
        return self.call('GET', url, user_ticket)

    def get_documents(self, uid, user_ticket, passenger_id):
        url = 'v1/travelers/{}/passengers/{}/documents'.format(uid, passenger_id)
        return self.call('GET', url, user_ticket)

    def get_traveler_documents(self, uid, user_ticket, fields):
        url = 'v1/travelers/{}?fields={}'.format(uid, ','.join(fields))
        return self.call('GET', url, user_ticket)

    def get_document_types(self):
        return self.call('GET', 'v1/document_types')
