# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

"""
https://st.yandex-team.ru/SPTNK-40
https://st.yandex-team.ru/RASPFRONT-2719
"""

import logging
from itertools import chain

import requests
import six
from django.conf import settings
from django.utils.encoding import force_text
from django.utils.functional import cached_property
from marshmallow import Schema, fields, pre_load, post_load
from pybreaker import CircuitBreaker
from requests.exceptions import ConnectionError, Timeout, HTTPError
from ylog.context import log_context

from common.data_api.yandex_bus import config
from common.dynamic_settings.core import DynamicSetting
from common.dynamic_settings.default import conf
from common.models.geo import Settlement, Station
from common.models.transport import TransportType
from common.models_utils.geo import Point
from common.serialization.common_schemas import PriceSchema
from common.serialization.schema import get_defaults_from_schema
from common.settings.utils import define_setting
from common.utils.namedtuple import namedtuple_with_defaults
from travel.library.python.tracing.instrumentation import traced_function

log = logging.getLogger(__name__)

define_setting('YANDEX_BUS_BREAKER_PARAMS', default={'fail_max': 3, 'reset_timeout': 60})

yandex_bus_breaker = CircuitBreaker(**settings.YANDEX_BUS_BREAKER_PARAMS)

SEARCH_TEMPLATE = '{url}/api/search/rasp?from-id={point_from}&to-id={point_to}&date={date}'
DATE_FORMAT = '%Y-%m-%d'

AVAILABLE_STATUS_ID = 1
EMPTY_NUMBER = '000'

conf.register_settings(YBUS_DONT_ASK_BETWEEN_SAME_POINT=DynamicSetting(
    False,
    cache_time=3 * 60,
    description='Не запрашивать данные у Я.Автобусов, если пункты отправления и прибытия находятся в одном городе, '
                'или совпадают'
))


@six.python_2_unicode_compatible
class BusApiError(Exception):
    def __init__(self, code, error):
        if not error:
            error = {}
        self.error = error
        self.code = code

    def __str__(self):
        return '{}({}): {}'.format(self.__class__, self.code, self.error)


@six.python_2_unicode_compatible
class BusConnectionError(Exception):
    def __init__(self, exception):
        self.root = exception

    def __str__(self):
        return '{}: {}'.format(self.__class__, repr(self.root))


@yandex_bus_breaker
@traced_function
def get_yandex_buses_results(point_from, point_to, date, settlement_keys=True):
    """
    Получить сегменты с ценами от Яндекс.Автобусы.
    АПИ Автобусов может отвечать:
      - 200 если все данные есть в кэше;
      - 206 если часть данных есть в кэше;
      - 202 если в кэше нет данных;
    Если по техническим причинам не удалось поместить задачу в асинхронную очередь,
    АПИ ответит синхронно с кодом ответа 200.

    :param point_from: откуда
    :type point_from: Point
    :param point_to: куда
    :type point_to: Point
    :param date: дата во временной зоне пункта отправления
    :type date: date
    :param settlement_keys: искать города вместо станций
    :type settlement_keys: bool
    :return:
    """
    if settlement_keys:
        point_from_req = getattr(point_from, 'settlement', None) or point_from
        point_to_req = getattr(point_to, 'settlement', None) or point_to
    else:
        point_from_req = point_from
        point_to_req = point_to

    if conf.YBUS_DONT_ASK_BETWEEN_SAME_POINT:
        if point_from_req == point_to_req:
            return [], False

    request_url = SEARCH_TEMPLATE.format(url=settings.YANDEX_BUS_API_URL,
                                         point_from=point_from_req.point_key, point_to=point_to_req.point_key,
                                         date=date.strftime(DATE_FORMAT))

    try:
        response = requests.get(request_url, timeout=config.YANDEX_BUS_API_TIMEOUT)
    except (ConnectionError, Timeout) as e:
        raise BusConnectionError(e)

    log.info('request_url {}, status_code {}'.format(response.request.url, response.status_code))
    if response.status_code in (429, 404):
        return [], False

    try:
        response.raise_for_status()
    except HTTPError:
        raise BusApiError(response.status_code, response.text)

    data = YandexBusSegmentSchema(strict=True).load(response.json(), many=True).data
    querying = response.status_code in [202, 206]

    segments = _filter_from_stations(data)

    return [segment for segment in segments
            if (_match(segment.station_from, point_from, point_from_req) and
                _match(segment.station_to, point_to, point_to_req)
                )
            ], querying


def _filter_from_stations(data):
    """
    https://st.yandex-team.ru/RASPFRONT-6631
    Оставляем только автобусы от станций, убираем автобусы от городов
    """
    segments = []
    for segment in data:
        from_key = segment.station_from.point_key
        to_key = segment.station_to.point_key
        from_cls, from_id = Point.parse_key(from_key)
        if from_cls == Station:
            segments.append(segment)
        else:
            log_msg = 'Yandex bus station key missed, segment_id: {}, from_key: {}, to_key: {}'.format(
                segment.id, from_key, to_key
            )

            with log_context(mess_type='Yandex bus station key missed',
                             segment_id=segment.id, from_key=from_key, to_key=to_key):
                log.warning(log_msg)
    return segments


def _match(station, point, point_req):
    return point == point_req or station.point_key == point.point_key or station.point_key == point_req.point_key


class StationAdapter(object):
    """Необходим чтобы адаптировать settlement для использования его как station.
    """
    _attrs = {
        'L_popular_title': 'L_title',
        'settlement_id': 'id',
    }

    def __init__(self, point):
        self._point = point

    def __eq__(self, other):
        return self._point == other._point

    def __getattr__(self, attr):
        try:
            return getattr(self._point, attr)
        except AttributeError:
            if attr in self._attrs:
                return getattr(self._point, self._attrs[attr])
            raise

    @property
    def settlement(self):
        return getattr(self._point, 'settlement', self._point)


class YandexBusSegmentSchema(Schema):
    id = fields.String()
    departure = fields.DateTime()
    arrival = fields.DateTime(allow_none=True)

    transport_model = fields.String(allow_none=True, load_from='bus')
    company_name = fields.String(allow_none=True, load_from='carrier')
    order_url = fields.String(load_from='book_url')

    number = fields.Function(deserialize=lambda value: force_text(value), allow_none=True)
    title = fields.String(allow_none=True, load_from='name')

    price = fields.Nested(PriceSchema)
    seats = fields.Integer(load_from='freeSeats')
    can_pay_offline = fields.Boolean(load_from='canPayOffline')

    station_from = fields.Method(deserialize='point_by_key', load_from='from')
    station_to = fields.Method(deserialize='point_by_key', load_from='to')

    @pre_load(pass_many=True)
    def prepare(self, data, many):
        if many:
            data = self.filter_data(data)
            self.collect_points(data)
        else:
            self.collect_points([data])
        return data

    @pre_load
    def add_price(self, data):
        data['price'] = {
            'value': data['price'],
            'currency': data['currency']
        }
        return data

    def collect_points(self, segments):
        settlement_ids = set()
        station_ids = set()
        for segment in segments:
            for fname in ('from', 'to'):
                point_key = segment.get(fname)
                if not point_key:
                    continue
                cls, pk = Point.parse_key(point_key)
                if cls == Settlement:
                    settlement_ids.add(int(pk))
                elif cls == Station:
                    station_ids.add(int(pk))
        settlements = Settlement.objects.in_bulk(list(settlement_ids))
        stations = Station.objects.in_bulk(list(station_ids))
        self.context['points'] = {
            p.point_key: StationAdapter(p)
            for p in chain(six.itervalues(settlements), six.itervalues(stations))
        }

    def point_by_key(self, key):
        if not self.context:
            return
        return self.context['points'].get(key)

    @staticmethod
    def filter_data(data):
        return [
            item for item in data
            if item['status']['id'] == AVAILABLE_STATUS_ID
        ]

    @post_load(pass_many=True)
    def filter_segments(self, data, many):
        """
        Если станции отправления или прибытия сегмента нет в нашей базе - отбрасываем его
        """

        def valid(segment):
            return segment['station_from'] is not None and segment['station_to'] is not None

        if many:
            return [s for s in data if valid(s)]
        else:
            return data if valid(data) else None

    @post_load
    def create_segment(self, data):
        data['departure'] = data['station_from'].localize(loc=data['departure'])
        if data.get('arrival'):
            data['arrival'] = data['station_to'].localize(loc=data['arrival'])
        if data.get('number') == EMPTY_NUMBER:
            data['number'] = None
        return Segment(**data)


class Segment(namedtuple_with_defaults('Segment',
                                       YandexBusSegmentSchema().fields.keys(),
                                       get_defaults_from_schema(YandexBusSegmentSchema))):
    @cached_property
    def duration(self):
        if not self.arrival:
            return None
        return self.arrival - self.departure

    def get_duration(self):
        duration = self.duration
        if duration is None:
            return None
        return int(duration.total_seconds() // 60)

    @cached_property
    def t_type(self):
        return TransportType.objects.get(pk=TransportType.BUS_ID)
