# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

import requests
from django.conf import settings

from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from common.utils.lxmlutils import get_sub_tag_text
from common.utils.safe_xml_parser import safe_xml_fromstring


log = logging.getLogger(__name__)


define_setting('YASMS_SENDER', default='rasp')
define_setting('YASMS_URL', {
    Configuration.PRODUCTION: 'https://sms.passport.yandex.ru',
    Configuration.TESTING: 'https://phone-passport-test.yandex.ru',
}, default=None)
define_setting('YASMS_DONT_SEND_ANYTHING', default=settings.YASMS_URL is None)


class YaSMSClientError(Exception):
    def __init__(self, err_code, err_msg):
        self.err_code = err_code
        self.err_msg = err_msg

    def __str__(self):
        return 'Error {}: {}'.format(self.err_code, self.err_msg)


class YaSMSClient(object):
    """
    Клиент для АПИ YaSMS
    https://doc.yandex-team.ru/Passport/YaSMSDevGuide/concepts/YaSMS_Main.html
    https://st.yandex-team.ru/RASPFRONT-3117

    client = YaSMSClient('https://phone-passport-test.yandex.ru', 'rasp')
    client.sendsms(u'wowsuchsms', phone=u'+71234567890')

    Аналогично curl:
    curl "https://phone-passport-test.yandex.ru/sendsms?sender=rasp&text=wowsuchsms&phone=+7424242"
    """

    def __init__(self, url=None, service_id=None):
        self.url = url or settings.YASMS_URL
        self.sender = service_id or settings.YASMS_SENDER

    def call_api(self, method, params=None):
        params = dict(params) if params else {}
        params['sender'] = self.sender

        url = '{}/{}'.format(self.url, method)
        response = requests.get(url, params=params, verify=False)
        response.raise_for_status()  # документированные ошибки типа "Bad phone number" приходят с кодом 200

        response_xml = safe_xml_fromstring(response.content)
        err_code = get_sub_tag_text(response_xml, 'errorcode', None)
        if err_code:
            err_msg = get_sub_tag_text(response_xml, 'error', None)
            raise YaSMSClientError(err_code, err_msg)

        return response_xml

    def sendsms(self, text, phone=None, uid=None):
        """
        Отправка смс через метод sendsms
        https://doc.yandex-team.ru/Passport/YaSMSDevGuide/reference/sendsms.html

        Дока не полностью актуальна, детали отправки смс по привязанным телефонам:
        https://wiki.yandex-team.ru/passport/yasms-to-blackbox-migration/
        """

        if phone:
            if uid:
                raise ValueError("Phone and uid args can't be defined together")

            param, value = 'phone', phone
        elif uid:
            param, value = 'uid', uid
        else:
            raise ValueError('Phone or uid must be defined')

        log.info('Посылаем SMS %s=%s, "%s"', param, value, text)
        if settings.YASMS_DONT_SEND_ANYTHING:
            response_xml = safe_xml_fromstring(b'''\
<?xml version="1.0" encoding="utf-8"?>
<doc>
    <message-sent id="log_only" />
</doc>
            ''')
        else:
            response_xml = self.call_api('sendsms', {
                'utf8': 1,
                'text': text,
                param: value,
            })

        return response_xml.xpath('message-sent')[0].get('id')
