# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import warnings
from functools import partial

from django.utils.encoding import force_text
from pymongo import UpdateOne, UpdateMany, DeleteOne, DeleteMany, InsertOne, ReplaceOne
from pymongo.errors import BulkWriteError

from travel.rasp.library.python.common23.logging import log_run_time
from travel.rasp.library.python.common23.utils.warnings import RaspDeprecationWarning


log = logging.getLogger(__name__)


class BulkBuffer(object):
    """
    Контекст-менеджер, позволяющий для заданной коллекции собирать операции записи в пачки
    заданного размера и отправлять их на сервер за раз.

    with BulkBuffer(my_collection) as coll:
        # апдейты будут выполнены пачкой при достижении max_buffer_size,
        # либо при выходе из with
        for item in some_list:
            coll.update(...)

    Работает только для пачек одинаковых операций - например, пачка update или пачка delete.
    Пачка разных операций будет уже на уровне pymongo разбита на отдельные пачки.
    """

    operations = {
        'update': UpdateOne,
        'update_one': UpdateOne,
        'replace_one': ReplaceOne,
        'update_many': UpdateMany,
        'insert_one': InsertOne,
        'delete_one': DeleteOne,
        'delete_many': DeleteMany,
    }

    def __init__(self, collection, max_buffer_size=10 ** 5, logger=None, on_before_flush=None):
        self.collection = collection
        self.max_buffer_size = max_buffer_size
        self.operations_processed = 0
        self.logger = logger or log
        self.on_before_flush = on_before_flush

        self.buffer = []

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.flush()

        self.logger.info('Total operations processed %s', self.operations_processed)

    def __getattr__(self, item):
        operation = self.operations.get(item)
        if operation:
            if item == 'update':
                warnings.warn(
                    "[2017-02-09] 'update' is deprecated in pymongo, use update_one or update_many",
                    RaspDeprecationWarning, stacklevel=2
                )

            return partial(self._run_operation, operation)

        super(BulkBuffer, self).__getattribute__(item)

    def _run_operation(self, operation, *args, **kwargs):
        self.buffer.append(operation(*args, **kwargs))
        if len(self.buffer) >= self.max_buffer_size:
            self.flush()

    def flush(self):
        msg = 'process {} operations on {}'.format(
            len(self.buffer), self.collection.full_name)

        with log_run_time(msg, logger=self.logger):
            if self.on_before_flush:
                self.on_before_flush(self)

            if self.buffer:
                try:
                    self.collection.bulk_write(self.buffer)
                except BulkWriteError as ex:
                    msg = '{}: {}'.format(force_text(ex), '; '.join(err['errmsg'] for err in ex.details['writeErrors']))
                    self.logger.exception(msg)
                    raise
                self.operations_processed += len(self.buffer)
                self.buffer = []
