# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from django.conf import settings
from django.db import connections, connection


def get_django_db_info():
    from django.conf import settings

    db_infos = []
    for db_alias, db in settings.DATABASES.items():
        db_info = {'db_alias': db_alias}
        db_infos.append(db_info)
        msg = '{}: '.format(db_alias)

        try:
            alias_getter = db.get('ALIAS_GETTER')
            db_wrapper = connections[db_alias]
            if alias_getter:
                db_info['alias_getter'] = alias_getter
                db_info['current_target'] = db_wrapper._get_alias()
                msg += 'alias proxy: {alias_getter} -> {current_target}'.format(**db_info)
            else:
                if getattr(db_wrapper, 'get_cluster', None):
                    cluster = db_wrapper.get_cluster()
                    host_or_cluster = repr(cluster)
                else:
                    host_or_cluster = db.get('HOST')

                msg += '{}'.format(host_or_cluster)
                port = db.get('PORT')
                if port:
                    msg += ':{}'.format(port)

                db_name = db.get('NAME')
                user = db.get('USER')
                msg += '; db "{}"; user "{}"'.format(db_name, user)
        except Exception as ex:
            msg += "unable to get info: {}".format(repr(ex))

        db_info['msg'] = msg

    return sorted(db_infos, key=lambda d: d['db_alias'])


def get_current_connection_info():
    cur = connection.cursor()
    cur.execute('select @@hostname;')
    res = list(cur.fetchall())
    return res[0][0]


def get_mysql_info():
    django_db_info = get_django_db_info()

    try:
        current_conn_host = get_current_connection_info()
    except Exception as ex:
        current_conn_host = 'unable to get conn: {}'.format(repr(ex))

    res = {
        'django_databases': django_db_info,
        'current_connection_host': current_conn_host,
        'current_dc': settings.YANDEX_DATA_CENTER
    }

    return res


RENDER_TEMPLATE = """
{databases}

Current connection: {current_connection_host}
Current dc: {current_dc}
"""

def render_mysql_info():
    mysql_info = get_mysql_info()

    return RENDER_TEMPLATE.format(
        databases='\n'.join(db['msg'] for db in mysql_info['django_databases']),
        **mysql_info
    )
