# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import re
from collections import namedtuple
from importlib import import_module

import six
from apispec import Path
from apispec.utils import load_operations_from_docstring, load_yaml_from_docstring
from django.conf import settings
from django.contrib.admindocs.views import simplify_regex
from django.core.urlresolvers import RegexURLPattern, RegexURLResolver
from rest_framework.views import APIView


Endpoint = namedtuple('Endpoint', ('path', 'method', 'callback'))


class EndpointEnumerator(object):
    """
    A class to determine the available API endpoints that a project exposes.
    Адаптировано отсюда:
        https://github.com/encode/django-rest-framework/blob/master/rest_framework/schemas/generators.py
    EndpointEnumerator (как и SchemaGenerator) появился в версии rest_framework 3.7+,
    а эта версия требует django-1.10+
    TODO: после апгрейда django и rest_framework - от этих утилит нужно избавиться.
    """
    _PATH_PARAMETER_COMPONENT_RE = re.compile(
        r'<(?:(?P<converter>[^>:]+):)?(?P<parameter>\w+)>'
    )

    def __init__(self, patterns=None, urlconf=None):
        if patterns is None:
            if urlconf is None:
                # Use the default Django URL conf
                urlconf = settings.ROOT_URLCONF

            # Load the given URLconf module
            if isinstance(urlconf, six.string_types):
                urls = import_module(urlconf)
            else:
                urls = urlconf
            patterns = urls.urlpatterns

        self.patterns = patterns

    def get_api_endpoints(self, patterns=None, prefix=''):
        """
        Return a list of all available API endpoints by inspecting the URL conf.
        """
        if patterns is None:
            patterns = self.patterns

        api_endpoints = []

        for pattern in patterns:
            path_regex = prefix + pattern.regex.pattern
            if isinstance(pattern, RegexURLPattern):
                path = self._get_path_from_regex(path_regex)
                callback = pattern.callback
                if self._should_include_endpoint(path, callback):
                    for method in self._get_allowed_methods(callback):
                        endpoint = Endpoint(path, method, callback)
                        api_endpoints.append(endpoint)

            elif isinstance(pattern, RegexURLResolver):
                nested_endpoints = self.get_api_endpoints(
                    patterns=pattern.url_patterns,
                    prefix=path_regex
                )
                api_endpoints.extend(nested_endpoints)

        api_endpoints = sorted(api_endpoints, key=self._endpoint_ordering)

        return api_endpoints

    def _get_path_from_regex(self, path_regex):
        """
        Given a URL conf regex, return a URI template string.
        """
        path = simplify_regex(path_regex)

        # Strip Django 2.0 convertors as they are incompatible with uritemplate format
        path = re.sub(self._PATH_PARAMETER_COMPONENT_RE, r'{\g<parameter>}', path)
        return path

    def _should_include_endpoint(self, path, callback):
        """
        Return `True` if the given endpoint should be included.
        """
        if not self._is_api_view(callback):
            return False  # Ignore anything except REST framework views.

        if hasattr(callback.cls, 'exclude_from_schema'):
            if getattr(callback.cls, 'exclude_from_schema', False):
                return False

        if path.endswith('.{format}') or path.endswith('.{format}/'):
            return False  # Ignore .json style URLs.

        return True

    def _get_allowed_methods(self, callback):
        """
        Return a list of the valid HTTP methods for this endpoint.
        """
        if hasattr(callback, 'actions'):
            actions = set(callback.actions.keys())
            http_method_names = set(callback.cls.http_method_names)
            methods = [method.upper() for method in actions & http_method_names]
        else:
            methods = callback.cls().allowed_methods

        return [method for method in methods if method not in ('OPTIONS', 'HEAD')]

    @staticmethod
    def _is_api_view(callback):
        """
        Return `True` if the given view callback is a REST framework view/viewset.
        """
        cls = getattr(callback, 'cls', None)
        return (cls is not None) and issubclass(cls, APIView)

    @staticmethod
    def _endpoint_ordering(endpoint):
        path, method, callback = endpoint
        method_priority = {
            'GET': 0,
            'POST': 1,
            'PUT': 2,
            'PATCH': 3,
            'DELETE': 4
        }.get(method, 5)
        return path, method_priority


def path_from_endpoint(endpoint, **kwargs):
    docstring = endpoint.callback.__doc__
    operations = load_operations_from_docstring(docstring) or {}
    if endpoint.method.lower() not in operations:
        docstring_yaml = load_yaml_from_docstring(docstring)
        if docstring_yaml:
            operations[endpoint.method.lower()] = {
                k: v for k, v in docstring_yaml.items() if k not in operations
            }
    return Path(path=endpoint.path, operations=operations)
