# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json

from django.core.serializers.json import DjangoJSONEncoder
from django.utils.encoding import force_bytes
from enum import Enum
from rest_framework import status
from rest_framework.permissions import AllowAny
from rest_framework.renderers import JSONRenderer
from rest_framework.response import Response
from rest_framework.views import APIView
from rest_framework_swagger import renderers


class _OpenAPIRenderer(renderers.OpenAPIRenderer):
    def render(self, data, accepted_media_type=None, renderer_context=None):
        if renderer_context['response'].status_code != status.HTTP_200_OK:
            return JSONRenderer().render(data)
        return force_bytes(json.dumps(data, ensure_ascii=False, cls=AnyJsonEncoder))


class AnyJsonEncoder(DjangoJSONEncoder):
    def default(self, o):
        try:
            if isinstance(o, Enum):
                return o.value
            return super(DjangoJSONEncoder, self).default(o)
        except TypeError:
            return repr(o)


def get_swagger_view(spec):
    class SwaggerSchemaView(APIView):
        _ignore_model_permissions = True
        exclude_from_schema = True
        permission_classes = [AllowAny]
        renderer_classes = [
            _OpenAPIRenderer,
            renderers.SwaggerUIRenderer
        ]

        def get(self, request):
            return Response(spec.to_dict())
    return SwaggerSchemaView.as_view()
