# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.conf import settings
from django.utils.module_loading import import_string
from mongoengine import Document, EmbeddedDocument, fields

from common.data_api.sendr.api import Attachment
from travel.rasp.library.python.common23.date.environment import now_utc


class ImproperConfigureError(Exception):
    pass


class AttachmentInfo(EmbeddedDocument):
    filename = fields.StringField(required=True)
    mime_type = fields.StringField(required=True)
    content = fields.BinaryField(required=True)

    def as_sendr_attachment(self):
        return Attachment(self.filename, self.mime_type, self.content)


class EmailIntent(Document):
    meta = {
        'db_alias': settings.EMAIL_SENDER_DB_ALIAS,
        'strict': False,
        'indexes': [
            'is_sent',
            'created_at',
            'max_time_in_queue_exceeded',
            ('created_at', 'is_sent', 'max_time_in_queue_exceeded')
        ],
        'index_background': True
    }

    key = fields.StringField(required=True, unique=True)

    email = fields.StringField(required=True)
    args = fields.DictField(required=True)
    attachments = fields.EmbeddedDocumentListField(AttachmentInfo, default=list)
    campaign_code = fields.StringField(required=True)

    is_sent = fields.BooleanField(required=True, default=False)
    created_at = fields.DateTimeField(required=True, default=now_utc)
    max_time_in_queue_exceeded = fields.BooleanField(default=False)

    data = fields.DictField(default=dict)
    log_context = fields.DictField(default=dict)
    preprocessor = fields.StringField(null=True)

    def _preprocessor_func(self):
        if not self.preprocessor:
            return None

        try:
            return import_string(self.preprocessor)
        except ImportError:
            raise ImproperConfigureError("Can't import preprocessor by path {}".format(self.preprocessor))

    def to_sendr_kwargs(self):
        if self.preprocessor:
            self._preprocessor_func()(self)
        return {
            'to_email': self.email,
            'args': self.args,
            'attachments': [
                attachment.as_sendr_attachment() for attachment in self.attachments
            ]
        }
