# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from mongoengine import NotUniqueError
from six import string_types

from .models import EmailIntent
from .tasks import send_email

log = logging.getLogger(__name__)


def guaranteed_send_email(key, to_email, args, campaign, attachments=None, data=None,
                          log_context=None, preprocessor=None):
    """
    Позволяет гарантированно отправить письмо.

    Пример использования:

      from common.email_sender import guaranteed_send_email, AttachmentInfo


      def attachment_adder(email):
          email.attachments.append(AttachmentInfo(
              filename='note.txt',
              mime_type='application/text',
              content='ZagZag!'
          ))

      guaranteed_send_email(
          key='my-unique-key',
          to_email='losos@example.org',
          args={'foo': 'bar'},
          campaign='YandexSellCampaign',
          data={'order_uid': '1'},
          log_context={'order_uid': '1'},
          preprocessor=attachment_adder
      )

    :param key: уникальный ключ письма, чтобы избежать повторной отправки
    :param to_email: кому шлем
    :param args: аргументы для рендера шаблона в sendr
    :param campaign: кампания
    :param attachments: вложения (AttachmentInfo)
    :param data: опциональные данные, будут включены в контекст логгирования, могут быть использованы в препроцессоре
    :param log_context: контекст логирования
    :param preprocessor: функция с сигнатурой (Email) -> Anything.
    :return:
    """
    if preprocessor is not None:
        if not isinstance(preprocessor, string_types):
            preprocessor = '{}.{}'.format(preprocessor.__module__, preprocessor.__name__)
    email = EmailIntent(
        key=key, email=to_email, args=args, campaign_code=campaign, data=data,
        log_context=log_context, preprocessor=preprocessor
    )
    if attachments:
        email.attachments = attachments

    try:
        email.save()
    except NotUniqueError:
        log.info('email с ключом %s для адресата %s с аргументами %s уже запланировано', key, to_email, args)
        return None
    else:
        send_email.apply_async([str(email.id)])
        return email.id
