# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from bson import ObjectId
from celery import shared_task
from django.conf import settings
from django.utils.module_loading import import_string
from ylog.context import log_context

from common.celery.task import single_launch_task
from common.data_api.sendr.api import Campaign
from travel.rasp.library.python.common23.date.environment import now_utc
from common.utils.metrics import report_progress

from .models import EmailIntent

log = logging.getLogger(__name__)


@shared_task()
def send_email(email_id, suppress_exceptions=False):
    email = None
    try:
        if not isinstance(email_id, ObjectId):
            email_id = ObjectId(email_id)
        email = EmailIntent.objects(id=email_id).get()
        with log_context(**email.log_context or {}):
            # Все равно, есть вероятность, что отправится больше 1 письма, если запустить эту функцию параллельно.
            if email.is_sent:
                log.info('Письмо %s уже отправлено', email_id)
                return
            if email.max_time_in_queue_exceeded:
                log.info('Письмо %s находится в очереди слишком долго, пропускаем его', email_id)
                return

            log.info('Начинаем отправку письма %s', email_id)
            campaign = Campaign.create_rasp_campaign(email.campaign_code)
            campaign.send(**email.to_sendr_kwargs())
            email.modify(is_sent=True)
            log.info('Успешно отправили письмо %s', email_id)
    except Exception:
        with log_context(**email and email.log_context or {}):
            log.exception('Не получилось отправить письмо с id %s', email_id)
        if email is not None:
            run_callback_if_max_time_in_queue_exceeded(email)
        if not suppress_exceptions:
            raise


def run_callback_if_max_time_in_queue_exceeded(email_intent):
    if now_utc() - email_intent.created_at > settings.EMAIL_SENDER_MAX_ALLOWED_TIME_IN_QUEUE:
        email_intent.modify(max_time_in_queue_exceeded=True)
        try:
            callback = settings.EMAIL_SENDER_CALLBACK_ON_MAX_ALLOWED_TIME_IN_QUEUE_EXCEEDED
            if callable(callback):
                callback(email_intent)
            else:
                import_string(callback)(email_intent)
        except Exception:
            log.exception('Не получилось выполнить действие с просроченным письмом id %s', email_intent.id)


@single_launch_task()
@report_progress('run_queue')
def run_queue():
    emails_in_queue = EmailIntent.objects(
        is_sent=False,
        created_at__lte=now_utc() - settings.EMAIL_SENDER_WAIT_INTERVAL,
        max_time_in_queue_exceeded=False
    )
    emails_in_queue_count = emails_in_queue.count()
    log.info('Обрабатываем %s из %s неотправленных писем.',
             min(emails_in_queue_count, settings.EMAIL_SENDER_NUMBER_TO_PROCESS_PER_LAUNCH),
             emails_in_queue_count)

    emails_to_process = emails_in_queue.order_by('-created_at')[:settings.EMAIL_SENDER_NUMBER_TO_PROCESS_PER_LAUNCH]
    for email in emails_to_process:
        if settings.EMAIL_SENDER_CONCURRENT_PROCESSING:
            send_email.apply_async([str(email.id), True])
        else:
            send_email(email.id, True)
