# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from django import forms
from django.core.exceptions import ObjectDoesNotExist, ValidationError

from common.models.geo import Station, Point
from common.models.transport import TransportType
from common.utils.date import fromSQLDate, fromSQLdatetime


class RouteForm(forms.Form):
    number = forms.CharField(required=True)
    plane = forms.BooleanField(required=False)
    train = forms.BooleanField(required=False)
    suburban = forms.BooleanField(required=False)
    bus = forms.BooleanField(required=False)

    def __init__(self, *args, **kwargs):
        kwargs['auto_id'] = "route_%s"
        super(RouteForm, self).__init__(*args, **kwargs)

    def clean_number(self):
        number = self.cleaned_data.get("number", '').strip()

        if not number:
            raise ValidationError(u"Введите номер маршрута.")

        return number

    def get_ttypes(self):
        ttypes = []

        for code in ('train', 'plane', 'suburban', 'bus'):
            if self.cleaned_data.get(code):
                ttypes.append(TransportType.objects.get(code=code).id)

        return ttypes


class ThreadForm(forms.Form):
    """ Форма для валидации GET параметров thread
        departure - дата старта нитки
        departure_from - дата/время отправления с указанной станции во временной зоне станции
    """
    departure = forms.CharField(required=False)
    departure_from = forms.CharField(required=False)
    station_from = forms.CharField(required=False)
    station_to = forms.CharField(required=False)

    point_from = forms.CharField(required=False)
    point_to = forms.CharField(required=False)

    def clean_departure(self):
        if 'departure' in self.cleaned_data and self.cleaned_data['departure']:
            try:
                return fromSQLDate(self.cleaned_data['departure'])
            except (ValueError, TypeError):
                pass
        return None

    def clean_departure_from(self):
        if 'departure_from' in self.cleaned_data and self.cleaned_data['departure_from']:
            try:
                return fromSQLdatetime(self.cleaned_data['departure_from'])
            except (ValueError, TypeError):
                pass
        return None

    def clean_station_from(self):
        try:
            return Station.objects.get(pk=self.cleaned_data['station_from'])

        except (Station.DoesNotExist, ValueError):
            return None

    def clean_station_to(self):
        try:
            return Station.objects.get(pk=self.cleaned_data['station_to'])

        except (Station.DoesNotExist, ValueError):
            return None

    def clean_point_from(self):
        if self.cleaned_data['point_from']:
            try:
                return Point.get_by_key(self.cleaned_data['point_from'])

            except (ValueError, ObjectDoesNotExist):
                pass

    def clean_point_to(self):
        if self.cleaned_data['point_to']:
            try:
                return Point.get_by_key(self.cleaned_data['point_to'])

            except (ValueError, ObjectDoesNotExist):
                pass
