# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

from django.conf import settings

from common.geotargeting import lookup
from common.models.geo import Settlement
from common.utils import request_helper
from common.utils.date import astimezone

log = logging.getLogger(__name__)


class Ip(object):
    """
    !!!Можно подключать в любом порядке, так как независимая
    """

    def process_request(self, request):
        request.client_ip = request_helper.get_ip(request)


class ClientCity(object):
    """
    Определяем:
    client_city_real - город пользователя
    default_city - город пользователя по умолчанию
    client_city - город пользователя в зависимости от нац версии
    geoid - id города в зависимости от нац версии
    client_time - в зависимости от client_city

    Зависимости от других middleware:
    1. Host
    2. Ip
    3. Now
    """
    def _get_client_city(self, request, ip):
        return lookup.get_city(request, ip)

    def _get_default_city(self, domain):
        return Settlement.get_default_for_root_domain(domain)

    def _get_city_domain(self, root_domain, city):
        if root_domain == 'com':
            return 'com'

        try:
            detected_country_domain = city.country.domain_zone or 'ru'
        except AttributeError:
            detected_country_domain = 'ru'

        # Авиабилеты не имеют домена kz, by и тп
        # Поэтому срабатывает не нужная замен города пользователя для ajax запросов
        allowed_domains = getattr(settings, 'ALLOWED_DOMAINS', None)
        if allowed_domains and detected_country_domain not in allowed_domains:
            detected_country_domain = 'ru'

        return detected_country_domain

    def _get_client_city_now(self, request, city):
        return astimezone(request.now, city)

    def process_request(self, request):
        ip = request.client_ip
        client_city_real, geoid = self._get_client_city(request, ip)
        default_city = self._get_default_city(request.root_domain)

        request.client_city_real = client_city_real
        request.geoid = geoid or default_city._geo_id
        request.client_city = client_city_real or default_city
        request.client_city_domain = self._get_city_domain(request.root_domain, request.client_city)
        request.default_city = default_city
        request.client_time = self._get_client_city_now(request, request.client_city)


class GeoTargeting(object):
    """
    Middelware для обратной совместимости, чтобы все нужные middleware были подключены.
    !!! Этот подход не очень хорош => лучше подключать только нужные middleware руками

    Например:
    1. Api - не нужна логика с Tune
    2. Почти всем сервисам не нужна логика с игнорированием этой middleware

    Зависимости от других Middleware:
    1. Все зависимости внутренних.
    """
    def __init__(self):
        self.ip_middleware = Ip()
        self.client_city_middleware = ClientCity()

    def process_request(self, request):
        request.client_city = None
        request.client_city_real = None

        self.ip_middleware.process_request(request)

        if request.path.startswith(settings.GEOTARGETING_EXCLUDE_PREFIXES):
            if not request.path.startswith(settings.GEOTARGETING_INCLUDE_PREFIXES):
                return

        self.client_city_middleware.process_request(request)
