# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import sys

from django.conf import settings
from django.core.management import BaseCommand, CommandError
from django.db import connection
from six.moves import input

from common.utils.admindumps import get_latest_dump_sandbox_url
from common.utils.dump import (
    database_exists, load_dump_url, load_dump_to_database, purge_database, update_database_from_dump_url,
    log as dump_log
)

log = logging.getLogger(__name__)


def confirm(message):
    response = input('{} (yes/no): '.format(message).encode('utf8'))

    while response not in ('yes', 'no'):
        response = input('Please enter either "yes" or "no": ')

    return response == 'yes'


class Command(BaseCommand):
    args = '<db_name>'
    help = u"Загружает свежий дамп в указанную базу"

    def add_arguments(self, parser):
        parser.add_argument(
            '--noinput',
            action='store_false', dest='interactive', default=True,
            help='Ничего не спрашивать в консоли'
        )
        parser.add_argument(
            '--db-type',
            choices=(settings.WORK_DB, settings.SERVICE_DB),
            help='База для получения дампа'
        )
        parser.add_argument(
            '--full',
            action='store_true',
            help='Не пропускать таблицы из LOAD_DB_EXCLUDE_TABLES'
        ),
        parser.add_argument(
            '-x', '--exclude-table',
            action='append', dest='exclude_tables',
            help='Пропускать таблицу'
        ),
        parser.add_argument(
            '--base-url',
            default=settings.LOAD_DB_BASE_URL,
            help='URL страницы с файлами дампов'
        ),
        parser.add_argument(
            '--dump-file',
            help='Локальный путь к файлу дампа'
        )

        parser.add_argument('db_name', type=str)

    def handle(self, db_name, db_type=None, **options):
        load_from_url = not bool(options['dump_file'])

        if int(options['verbosity']) > 1:
            handler = logging.StreamHandler(sys.stdout)
            log.addHandler(handler)
            dump_log.addHandler(handler)

        loaded_dump_url, latest_dump_url = None, None
        if load_from_url:
            latest_dump_url = get_latest_dump_sandbox_url(db_type)
            if not latest_dump_url:
                raise CommandError('Нет свежего дампа базы {}'.format(db_type))

            loaded_dump_url = load_dump_url(db_name)
            if loaded_dump_url == latest_dump_url:
                raise CommandError('Дамп {} уже загружен'.format(latest_dump_url))

        connection.ensure_connection()
        connection_host = connection.connection.conn_params['host']
        if options['interactive'] and (
            database_exists(db_name) and
            not confirm('Содержимое базы {} на {} будет удалено. Продолжить?'.format(db_name, connection_host))
        ):
            sys.exit(1)

        if load_from_url and loaded_dump_url:
            log.info('Затираем предыдущий загруженный дамп %s', loaded_dump_url)

        exclude_tables = [] if options['full'] else list(settings.LOAD_DB_EXCLUDE_TABLES)
        if options['exclude_tables']:
            exclude_tables.extend(options['exclude_tables'])

        if load_from_url:
            update_database_from_dump_url(
                db_name, latest_dump_url, exclude_tables=exclude_tables
            )
        else:
            with open(options['dump_file']) as dump_file:
                purge_database(db_name)
                load_dump_to_database(dump_file, db_name, exclude_tables)
