# -*- coding: utf-8 -*-

"""
Реализация мультидоменной куки yandexuid.
https://wiki.yandex-team.ru/Cookies/yandexuid – Описание формата и назначения куки yandexuid

За основу взяты:
- NodeJS-реализация: https://github.yandex-team.ru/project-stub/express-yandexuid
- Старая Python-реализация: https://github.yandex-team.ru/common-python/python-django-tune
"""

import random
import time

from common.utils.ya import has_valid_yandexuid

YANDEXUID_MAX_AGE = 315569520  # количество секунд в 10 годах


def gen_yandexuid():
    return str(random.randint(0, 1e9 - 1)) + str(int(time.time()))


def need_new_yandexuid(request):
    if has_valid_yandexuid(request):
        return False

    if not request.uatraits_result.get('isBrowser'):
        return False

    if request.uatraits_result.get('isRobot'):
        return False

    if request.is_ajax():
        return False

    return True


class YandexuidMiddleware(object):
    def process_request(self, request):
        request.yandexuid = request.COOKIES.get('yandexuid')
        if not need_new_yandexuid(request):
            return

        # Генерируем и запоминаем yandexuid. Будем использовать это значение в process_response().
        request.yandexuid = gen_yandexuid()
        request.has_new_yandexuid = True

    def process_response(self, request, response):
        if getattr(request, 'has_new_yandexuid', False):
            response.set_cookie(
                key='yandexuid',
                value=request.yandexuid,
                domain='.yandex.{tld}'.format(tld=request.tld),
                max_age=YANDEXUID_MAX_AGE,
                secure=True
            )
            del request.has_new_yandexuid
        return response
