# -*- coding: utf-8 -*-

import logging
import threading

from django.db import models
from django.utils import translation
from django.utils.translation import ugettext_lazy as _

from common.xgettext.i18n import xformat, XFormatParseError

log = logging.getLogger(__name__)

# Сюда middleware допишет аттрибут markstatic
local = threading.local()


class StaticText(models.Model):
    u""" Тексты дискламеров и сообщений об ошибках """
    code = models.CharField(verbose_name=_(u"код"), max_length=150, unique=True)
    title = models.CharField(verbose_name=_(u"название"), max_length=100,
                             help_text=_(u"отображается только в админке"))

    content = models.TextField(verbose_name=_(u"текст"))

    LANGUAGES = ['be', 'kk', 'ru', 'tt', 'uk', 'en', 'tr']
    LANGUAGES_TANKER = ['ru', 'en', 'uk', 'tr']

    announce_ru = models.TextField(verbose_name=_(u"анонс на русском"), null=True, blank=True)
    announce_be = models.TextField(verbose_name=_(u"анонс на белорусском"), null=True, blank=True)
    announce_kk = models.TextField(verbose_name=_(u"анонс на казахском"), null=True, blank=True)
    announce_tt = models.TextField(verbose_name=_(u"анонс на татарском"), null=True, blank=True)
    announce_uk = models.TextField(verbose_name=_(u"анонс на украинском"), null=True, blank=True)
    announce_en = models.TextField(verbose_name=_(u"анонс на английском"), null=True, blank=True)
    announce_tr = models.TextField(verbose_name=_(u"анонс на турецком"), null=True, blank=True)

    content_ru = models.TextField(verbose_name=_(u"текст на русском"), null=True, blank=True)
    content_be = models.TextField(verbose_name=_(u"текст на белорусском"), null=True, blank=True)
    content_kk = models.TextField(verbose_name=_(u"текст на казахском"), null=True, blank=True)
    content_tt = models.TextField(verbose_name=_(u"текст на татарском"), null=True, blank=True)
    content_uk = models.TextField(verbose_name=_(u"текст на украинском"), null=True, blank=True)
    content_en = models.TextField(verbose_name=_(u"текст на английском"), null=True, blank=True)
    content_tr = models.TextField(verbose_name=_(u"текст на турецком"), null=True, blank=True)

    sample_url = models.CharField(verbose_name=_(u"пример страницы"), max_length=100, blank=True,
                                  null=True, help_text=_(u"где используется"))
    is_mobile = models.BooleanField(verbose_name=_(u'для мобильной версии?'), default=False)

    objects = models.Manager()

    def __unicode__(self):
        return self.title

    class Meta:
        app_label = 'staticpages'
        verbose_name = _(u'текст')
        verbose_name_plural = _(u'тексты')

    @classmethod
    def get_text_by_code(cls, code, **kwargs):
        markstatic = getattr(local, 'markstatic', False)
        marktrans = getattr(translation, 'marktrans', False)

        try:
            text = cls.objects.get(code=code)

            field = 'content_%s' % translation.get_language()

            content = getattr(text, field, None) or text.content_ru

            if marktrans:
                content = '[dbtrans]%s[/dbtrans]' % content

            if markstatic:
                content = '[static code="%s"]%s[/static]' % (code, content)

        except (cls.DoesNotExist, TypeError):
            log.warning(u'Object %s not found by code %s', cls, repr(code))

            if markstatic:
                '[static code="%s" error="NOT_FOUND"/]' % code

            return ''

        if kwargs:
            return safe_xformat(code, content, kwargs)

        return content

    @classmethod
    def get_disclaimer_by_code(cls, code, **kwargs):
        markstatic = getattr(local, 'markstatic', False)
        marktrans = getattr(translation, 'marktrans', False)

        try:
            text = cls.objects.get(code=code)

            announce_field = 'announce_%s' % translation.get_language()
            field = 'content_%s' % translation.get_language()

            announce = getattr(text, announce_field, None) or text.announce_ru
            content = getattr(text, field, None) or text.content_ru

            if marktrans:
                announce = '[dbtrans]%s[/dbtrans]' % announce
                content = '[dbtrans]%s[/dbtrans]' % content

            if markstatic:
                announce = '[static code="%s"]%s[/static]' % (code, announce)
                content = '[static code="%s"]%s[/static]' % (code, content)

        except (cls.DoesNotExist, TypeError):
            log.warning(u'Object %s not found by code %s', cls, repr(code))

            if markstatic:
                '[static code="%s" error="NOT_FOUND"/]' % code

            return ''

        if kwargs:
            return safe_xformat(code, content, **kwargs)

        return {
            'announce': announce,
            'content': content
        }


def safe_xformat(code, content, kwargs):
    try:
        return xformat(content, **kwargs)
    except XFormatParseError:
        log.exception('Bad string for code %r: %r' % (code, content))
    except:
        log.exception('Error while formatting string for code %r: %r' % (code, content))

    return ''
