# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import six
from contextlib import contextmanager
from lxml import etree

from django.core.exceptions import ValidationError
from django.db import models
from django.db.models import Q
from django.db.models.signals import pre_save
from django.dispatch import receiver
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.models.core.geo.base_settlement_image import BaseSettlementImage  # noqa
from travel.rasp.library.python.common23.models.core.directions.direction import Direction  # noqa
from travel.rasp.library.python.common23.models.core.directions.direction_from_translate import DirectionFromTranslate  # noqa
from travel.rasp.library.python.common23.models.core.directions.direction_marker import DirectionMarker  # noqa
from travel.rasp.library.python.common23.models.core.directions.direction_translate import DirectionTranslate  # noqa
from travel.rasp.library.python.common23.models.core.directions.external_direction import ExternalDirection  # noqa
from travel.rasp.library.python.common23.models.core.directions.external_direction_marker import ExternalDirectionMarker  # noqa
from travel.rasp.library.python.common23.models.core.geo.base_station import BaseStation, StationProperty  # noqa
from travel.rasp.library.python.common23.models.core.geo.city_majority import CityMajority  # noqa
from travel.rasp.library.python.common23.models.core.geo.code_system import CodeSystem  # noqa
from travel.rasp.library.python.common23.models.core.geo.country import Country  # noqa
from travel.rasp.library.python.common23.models.core.geo.district import District  # noqa
from travel.rasp.library.python.common23.models.core.geo.point import Point  # noqa
from travel.rasp.library.python.common23.models.core.geo.point_synonym import PointSynonym  # noqa
from travel.rasp.library.python.common23.models.core.geo.region import Region  # noqa
from travel.rasp.library.python.common23.models.core.geo.replace_exception import  ReplaceException  # noqa
from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement, SettlementProperty  # noqa
from travel.rasp.library.python.common23.models.core.geo.station import Station  # noqa
from travel.rasp.library.python.common23.models.core.geo.station_code import StationCode  # noqa
from travel.rasp.library.python.common23.models.core.geo.station_code_manager import StationCodeManager  # noqa
from travel.rasp.library.python.common23.models.core.geo.station_majority import StationMajority  # noqa
from travel.rasp.library.python.common23.models.core.geo.station_phone import StationPhone  # noqa
from travel.rasp.library.python.common23.models.core.geo.station_terminal import StationTerminal  # noqa
from travel.rasp.library.python.common23.models.core.geo.station_type import StationType  # noqa
from travel.rasp.library.python.common23.models.core.geo.suburban_zone import SuburbanZone  # noqa
from travel.rasp.library.python.common23.models.core.geo.timezone_mixin import TimeZoneMixin  # noqa
from travel.rasp.library.python.common23.models.core.geo.translocal_mixin import TransLocalMixin  # noqa
from travel.rasp.library.python.common23.models.core.geo.way_to_airport import WayToAirport, WAY_TYPE_CHOICES  # noqa
from travel.rasp.library.python.common23.models.fields import CodeCharField, TrimmedCharField
from travel.rasp.library.python.common23.models.texts.geobase_ltitle import Geobase_L_title  # noqa
from travel.rasp.library.python.common23.models.utils.swap_model import SwapModel

log = logging.getLogger(__name__)


class SettlementRelatedStations(models.Model):
    """ Соответствия станций и городов """
    settlement = models.ForeignKey('Settlement', related_name='related_stations', verbose_name=_(u'город'))
    station = models.ForeignKey('Station', related_name='related_settlements', verbose_name=_(u'станция'))

    class Meta:
        db_table = 'www_settlement_related_stations'
        verbose_name = _(u'соответствие станции и города для отображения')
        verbose_name_plural = _(u'соответствия станций и городов для отображения')
        ordering = ('settlement', 'station')
        app_label = 'www'


class SettlementNearest(models.Model):
    """ Соответствия станций и городов """
    settlement = models.ForeignKey('Settlement', related_name='related_settlement', verbose_name=_(u'город'))
    nearest = models.ForeignKey('Settlement', related_name='related_nearest', verbose_name=_(u'ближайший'))
    description = models.TextField(verbose_name=_(u"описание проезда"), null=True, blank=True)

    class Meta:
        verbose_name = _(u'ближайший город')
        verbose_name_plural = _(u'ближайшие города')
        app_label = 'www'
        db_table = 'www_settlementnearest'


@receiver(pre_save, sender=Station)
def station_change_tablo_state_handler(sender, **kwargs):
    """
    RASP-13834 При смене на значение "Нет данных" нужно запоминать предыдущее значение и выводить его под селектом
    """

    instance = kwargs['instance']

    if not instance.id:
        return

    try:
        old_value = Station.objects.get(id=instance.id)

        if instance.tablo_state != old_value.tablo_state:
            instance.tablo_state_prev = old_value.tablo_state

    except Station.DoesNotExist:
        pass


class StationPassage(models.Model):
    u"""Пеший переход между станциями"""

    station_from = models.ForeignKey(Station, verbose_name=_(u'с какой станции'),
                                     null=False, blank=False,
                                     related_name='outgoing_passages')
    station_to = models.ForeignKey(Station, verbose_name=_(u'до какой станции'),
                                   null=False, blank=False,
                                   related_name='incoming_passages')
    terminal_from = models.ForeignKey(StationTerminal, verbose_name=_(u'с какого выхода'),
                                      null=True, blank=True, default=None,
                                      related_name='outgoing_passages')
    terminal_to = models.ForeignKey(StationTerminal, verbose_name=_(u'до какого выхода'),
                                    null=True, blank=True, default=None,
                                    related_name='incoming_passages')
    duration = models.IntegerField(verbose_name=_(u'длительность перехода'),
                                   null=True, blank=True, default=None)
    is_dual = models.BooleanField(verbose_name=_(u'переход двусторонний'),
                                  null=False, blank=False, default=True)

    def __unicode__(self):
        return "%s-%s" % (self.station_from.title, self.station_to.title)

    def as_xml(self, code_system=None):
        link = etree.Element('link',
                             time=str(self.duration),
                             bidirectional=str(int(self.is_dual)),
                             enabled='1')

        _from = etree.Element('from')
        _to = etree.Element('to')

        if code_system is not None:
            if self.station_from.get_code(code_system):
                _from.set('vendor', code_system)
                _from.set('vendor_id', self.station_from.get_code(code_system))
            else:
                _from.set('station_id', str(self.station_from_id))

            if self.station_to.get_code(code_system):
                _to.set('vendor', code_system)
                _to.set('vendor_id', self.station_to.get_code(code_system))
            else:
                _to.set('station_id', str(self.station_to_id))
        else:
            _from.set('station_id', str(self.station_from_id))
            _to.set('station_id', str(self.station_to_id))

        if self.terminal_from_id:
            _from.set('terminal', self.terminal_from.code)

        if self.terminal_to_id:
            _to.set('terminal', self.terminal_to.code)

        link.append(_from)
        link.append(_to)

        return link

    class Meta:
        verbose_name = _(u'переход между станциями')
        verbose_name_plural = _(u'переходы между станциями')
        app_label = 'www'
        db_table = 'www_stationpassage'


class Station2Settlement(models.Model):
    """ Соответствия станций и городов """

    station = models.ForeignKey(Station, verbose_name=_(u'станция'))
    settlement = models.ForeignKey(Settlement, verbose_name=_(u'город'))

    class Meta:
        verbose_name = _(u'соответствиe станции и города')
        verbose_name_plural = _(u'соответствия станций и городов')
        app_label = 'www'
        db_table = 'www_station2settlement'


class SettlementCode(models.Model):
    u"""Код города"""
    settlement = models.ForeignKey(Settlement, verbose_name=_(u'город'), blank=False,
                                   null=False, related_name='code_set')
    code = CodeCharField(verbose_name=_(u"код"), blank=False, null=False,
                         max_length=50, db_index=True)
    system = models.ForeignKey(CodeSystem, verbose_name=_(u'система кодирования'),
                               blank=False, null=False)

    def __unicode__(self):
        return self.settlement.title + u': ' + self.system.code + u' = ' + self.code

    class Meta:
        ordering = ('settlement__id',)
        verbose_name = _(u'код города')
        verbose_name_plural = _(u'коды городов')
        # В каждой системе кодирования город имеет только один код
        unique_together = (('settlement', 'system'), ('system', 'code'))
        app_label = 'www'
        db_table = 'www_settlementcode'


class StationExpressAlias(models.Model):
    u"""Название станции в системе Экспресс"""

    station = models.ForeignKey(
        Station, verbose_name=_(u'станция'), blank=False,
        null=False, related_name='expressalias_set'
    )

    alias = CodeCharField(
        _(u'название'),
        blank=False, null=False,
        max_length=50, db_index=True, unique=True
    )

    class Meta:
        app_label = 'www'
        db_table = 'www_stationexpressalias'
        verbose_name = _(u'название в системе Экспресс')


class PseudoRegion(models.Model):
    title = models.CharField(_(u"название"), max_length=255, null=False, blank=False)
    countries = models.ManyToManyField(Country, blank=True, verbose_name=_(u"страны"))

    class Meta:
        verbose_name = _(u"псевдо-регион")
        verbose_name_plural = _(u"псевдо-регионы")
        app_label = 'www'
        db_table = 'www_pseudoregion'

    def __unicode__(self):
        return self.title


PseudoRegion.countries.through._meta.verbose_name = _(u"страна псевдорегиона")
PseudoRegion.countries.through._meta.verbose_name_plural = _(u"страны псевдорегиона")


class GortransCityLink(models.Model):
    city = models.ForeignKey(Settlement, null=False, blank=False, verbose_name=_(u'город'))
    url = models.CharField(u"Ссылка", max_length=255, null=False, blank=False)

    def url_a(self):
        return '<a href="%s" target="_blank">%s</a>' % (self.url, self.url)

    url_a.allow_tags = True

    def __unicode__(self):
        return self.city.title

    class Meta:
        verbose_name = _(u"ссылка на гортранс")
        verbose_name_plural = _(u"ссылки на гортранс")
        app_label = 'www'
        db_table = 'www_gortranscitylink'


class CompanyOffice(models.Model):
    title = models.CharField(verbose_name=_(u'Название'), max_length=255)
    company = models.ForeignKey('www.Company', verbose_name=_(u'Компания перевозчик'))
    settlement = models.ForeignKey('www.Settlement', verbose_name=_(u'Город'))
    address = models.CharField(max_length=255, verbose_name=_(u'Адресс'))

    main_station = models.ForeignKey('www.Station', verbose_name=_(u'Главная станция'),
                                     null=True, blank=True, default=None)
    contact_info = models.TextField(verbose_name=_(u'Контактная информация'),
                                    null=False, blank=True, default='')
    phone = TrimmedCharField(verbose_name=_(u'Контактный телефон'), max_length=255,
                             null=False, blank=True, default='')
    phone_booking = TrimmedCharField(verbose_name=_(u'Телефон для бронирования'),
                                     max_length=255, blank=True, default='')
    description = models.TextField(verbose_name=_(u'Описание'), blank=True, default='')
    longitude = models.FloatField(verbose_name=_(u'Долгота'), null=True, blank=True, default=None)
    latitude = models.FloatField(verbose_name=_(u'Широта'), null=True, blank=True, default=None)
    is_main = models.BooleanField(verbose_name=_(u'Главный офис'), default=False)

    def __unicode__(self):
        return self.title

    def clean(self):
        # У перевозчика может быть только один главный офис
        if self.is_main and self.__class__.objects.filter(company=self.company, is_main=True).exists():
            raise ValidationError(_(u'У компании-перевозчика %s уже есть главый офис' % self.company))

    class Meta:
        verbose_name = _(u'Офис компании-перевозчика')
        verbose_name_plural = _(u'Офисы компаний-перевозчиков')
        app_label = 'www'


class SettlementImage(six.with_metaclass(SwapModel, BaseSettlementImage)):
    admin_module_path = 'api.models'

    class Meta(BaseSettlementImage.Meta):
        verbose_name = _(u'Изображение города')
        verbose_name_plural = _(u'Изображение городов')
        app_label = 'api'
        db_table = 'api_settlementimage'
