# coding: utf-8

from datetime import datetime

from django import forms
from django.conf import settings
from django.core.exceptions import MultipleObjectsReturned, ObjectDoesNotExist
from django.core.files.storage import get_storage_class
from django.core.validators import MaxValueValidator, MinValueValidator
from django.db import models
from django.utils.deconstruct import deconstructible
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _
from lxml import etree

from common.db.mds.s3_storage import mds_s3_media_storage
from common.models_utils.i18n import L_field
from common.utils.caching import cache_until_switch_thread_safe
from travel.rasp.library.python.common23.date.environment import get_request
from common.utils.fields import CodeCharField
from common.utils.mathematic import round_dispersed


if settings.ENVIRONMENT not in ['dev', 'testing', 'production']:
    raise EnvironmentError(u'Неверно указана среда выполнения')


MAX_UPLOADED_PARTNER_LOGO_SIZE_IN_BYTES = 1 * 1024 * 1024


class CleanImageMixin(object):
    def clean(self, *args, **kwargs):
        data = super(PartnerLogoImageField, self).clean(*args, **kwargs)

        if data.file:
            if data.file.size > MAX_UPLOADED_PARTNER_LOGO_SIZE_IN_BYTES:
                raise forms.ValidationError(
                    _(u'Файл %s превышает допустимый лимит %.2f МБ')
                    % (data.name, MAX_UPLOADED_PARTNER_LOGO_SIZE_IN_BYTES / 1024. / 1024.)
                )

        # Try to import PIL in either of the two ways it can end up installed.
        try:
            from PIL import Image

        except ImportError:
            import Image

        try:
            image = Image.open(data.file)

        except IOError:
            raise forms.ValidationError(_(u'Файл не найден %s') % data.name)

        image_format = image.format.lower()

        file_ext = settings.PARTNERS_LOGO_FORMATS.get(image_format, None)

        if not file_ext:
            raise forms.ValidationError(_('File type %s not supported.' % image_format))

        else:
            self.file_ext = file_ext

        return data


# TODO: PartnerLogoImageField и PartnerLogoSvgImageField должны быть в админке,
# в common их можно заменить на ImageField
# Но поля логотипов в Partner нужны в common,
# а переопределение полей модели в админке сейчас не работает
class PartnerLogoImageField(models.ImageField, CleanImageMixin):
    def __init__(self, *args, **kwargs):
        if 'upload_to' in kwargs:
            raise Exception('PartnerLogoImageField has custom upload_to!!!')

        kwargs['upload_to'] = self.generate_filename
        super(PartnerLogoImageField, self).__init__(*args, **kwargs)

    def deconstruct(self):
        name, path, args, kwargs = super(PartnerLogoImageField, self).deconstruct()

        del kwargs['upload_to']

        return name, path, args, kwargs

    def generate_filename(self, partner, filename):
        return 'data/partners/logo/%s' % '_'.join(
            filter(None, [
                partner.code,
                self.name.replace('logo', '').lstrip('_'),
            ])
        ) + '.' + self.file_ext


class Svg2PngImageField(models.ImageField, CleanImageMixin):
    pass


class PartnerLogoSvgImageField(models.FileField):
    def __init__(self, *args, **kwargs):
        if 'upload_to' in kwargs:
            raise Exception('PartnerLogoSvgImageField has custom upload_to!!!')

        kwargs['upload_to'] = self.generate_filename
        super(PartnerLogoSvgImageField, self). \
            __init__(*args, **kwargs)

    def deconstruct(self):
        name, path, args, kwargs = super(PartnerLogoSvgImageField, self).deconstruct()

        del kwargs['upload_to']

        return name, path, args, kwargs

    def clean(self, *args, **kwargs):
        data = super(PartnerLogoSvgImageField, self).clean(*args, **kwargs)

        if data.file:
            if data.file.size > MAX_UPLOADED_PARTNER_LOGO_SIZE_IN_BYTES:
                raise forms.ValidationError(
                    _(u'Файл %s превышает допустимый лимит %.2f МБ')
                    % (data.name, MAX_UPLOADED_PARTNER_LOGO_SIZE_IN_BYTES / 1024. / 1024.)
                )

        try:
            etree.parse(data.file)
        except IOError:
            raise forms.ValidationError(_(u'Файл не найден %s') % data.name)
        except Exception:
            raise forms.ValidationError(_(u'Формат файла не поддерживается'))

        return data

    def generate_filename(self, partner, filename):
        return 'data/partners/logo_svg/%s' % '_'.join(
            filter(None, [
                partner.code,
                self.name.replace('logo_svg', '').lstrip('_'),
            ])
        ) + '.svg'


@deconstructible
class PartnerLogoFileSystemStorage(get_storage_class()):
    def get_available_name(self, name):
        if self.exists(name):
            self.delete(name)

        return name


@deconstructible
class LogoSvg2PngUploadToGetter(object):
    def __init__(self, lang):
        self.lang = lang

    def __call__(self, instance, filename):
        return 'data/partners/svg2png/id{}_{}.png'.format(instance.id, self.lang)


class Partner(models.Model):
    L_title = L_field(_(u'Название партнера'), default=u'', max_length=100, blank=True,
                      add_local_field=True, local_field_critical=True,)

    L_national_ru_title = L_field(
        _(u'Название в русской версии'),
        default=u'', max_length=100, blank=True, base_lang='ru',
        fallback_fields=['L_title']
    )

    L_national_ua_title = L_field(
        _(u'Название в украинской версии'),
        default=u'', max_length=100, blank=True, base_lang='uk',
        fallback_fields=['L_title']
    )

    L_national_tr_title = L_field(
        _(u'Название в турецкой версии'),
        default=u'', max_length=100, blank=True, base_lang='tr',
        fallback_fields=['L_title']
    )

    L_national_com_title = L_field(
        _(u'Название в com версии'),
        default=u'', max_length=100, blank=True, base_lang='en',
        fallback_fields=['L_title']
    )

    site_url = models.CharField(verbose_name=_(u"сайт партнера"), max_length=255)

    code = CodeCharField(verbose_name=_(u'код партнера'),
                         max_length=100, blank=True, default=None,
                         unique=True, null=True)

    billing_datasource_id_dev = models.IntegerField(verbose_name=_(u'iD партнера в системе биллинга dev'),
                                                    blank=True, null=True, default=None, help_text=_(u"datasource ID"))

    billing_datasource_id_testing = models.IntegerField(verbose_name=_(u'iD партнера в системе биллинга testing'),
                                                        blank=True, null=True, default=None,
                                                        help_text=_(u"datasource ID"))

    billing_datasource_id_production = models.IntegerField(verbose_name=_(u'iD партнера в системе биллинга production'),
                                                           blank=True, null=True, default=None,
                                                           help_text=_(u"datasource ID"))

    click_price = models.FloatField(verbose_name=_(u'стоимость клика, у.е.'),
                                    blank=False, null=False)

    click_price_ru = models.FloatField(
        verbose_name=_(u'стоимость клика в Российской версии, у.е.'),
        blank=False, null=False
    )

    click_price_ua = models.FloatField(
        verbose_name=_(u'стоимость клика в Украинской версии, у.е.'),
        blank=False, null=False
    )

    click_price_tr = models.FloatField(
        verbose_name=_(u'стоимость клика в Турецкой версии, у.е.'),
        blank=False, null=False
    )

    click_price_com = models.FloatField(
        verbose_name=_(u'стоимость клика в com версии, у.е.'),
        blank=False, null=False
    )

    current_balance = models.FloatField(verbose_name=_(u'текущий баланс, у.е.'),
                                        blank=False, null=False, default=0.0)

    notify_balance_threshold = models.FloatField(verbose_name=_(u'порог баланса для уведомления, у.е.'),
                                                 blank=True, null=True, default=None)
    low_balance_notification_sent = models.DateTimeField(verbose_name=_(u"Уведомление о низком балансе послано"),
                                                         blank=False, null=False, default=datetime(2011, 1, 1))
    null_balance_notification_sent = models.DateTimeField(verbose_name=_(u"Уведомление о нулевом балансе послано"),
                                                          blank=False, null=False, default=datetime(2011, 1, 1))
    statistic_reminder_sent = models.DateTimeField(verbose_name=_(u"Напоминание про статистику продаж послано"),
                                                   blank=False, null=False, default=datetime(2011, 1, 1))

    balance_updated_at = models.DateTimeField(verbose_name=_(u'баланс обновлен'),
                                              blank=True, null=True, default=None)

    disabled = models.BooleanField(verbose_name=_(u'принудительно отключить партнера'),
                                   blank=False, null=False, default=False)

    foreign_currency = models.BooleanField(
        verbose_name=_(u'Иностранные валюты'),
        help_text=_(u'Импортировать предложения в валютах '
                    u'отличных от валюты национальной версии'),
        blank=False, null=False, default=False
    )

    enabled_in_ticket_ru = models.BooleanField(
        _(u'Запрашивать из КБ (ru)'), blank=False, null=False, default=False
    )

    enabled_in_ticket_ua = models.BooleanField(
        _(u'Запрашивать из КБ (ua)'), blank=False, null=False, default=False
    )

    enabled_in_ticket_tr = models.BooleanField(
        _(u'Запрашивать из КБ (tr)'), blank=False, null=False, default=False
    )

    enabled_in_ticket_com = models.BooleanField(
        _(u'Запрашивать из КБ (com)'), blank=False, null=False, default=False
    )

    enabled_in_rasp_ru = models.BooleanField(
        _(u'Запрашивать из ПМ (ru)'), blank=False, null=False, default=False
    )

    enabled_in_rasp_ua = models.BooleanField(
        _(u'Запрашивать из ПМ (ua)'), blank=False, null=False, default=False
    )

    enabled_in_rasp_tr = models.BooleanField(
        _(u'Запрашивать из ПМ (tr)'), blank=False, null=False, default=False
    )

    enabled_in_mobile_ticket_ru = models.BooleanField(
        _(u'Запрашивать на мобильных из КБ (ru)'), blank=False, null=False, default=False
    )

    enabled_in_mobile_ticket_ua = models.BooleanField(
        _(u'Запрашивать на мобильных из КБ (ua)'), blank=False, null=False, default=False
    )

    enabled_in_mobile_ticket_tr = models.BooleanField(
        _(u'Запрашивать на мобильных из КБ (tr)'), blank=False, null=False, default=False
    )

    enabled_in_mobile_ticket_com = models.BooleanField(
        _(u'Запрашивать на мобильных из КБ (com)'), blank=False, null=False, default=False
    )

    enabled_in_mobile_rasp_ru = models.BooleanField(
        _(u'Запрашивать на мобильных из ПМ (ru)'), blank=False, null=False, default=False
    )

    enabled_in_mobile_rasp_ua = models.BooleanField(
        _(u'Запрашивать на мобильных из ПМ (ua)'), blank=False, null=False, default=False
    )

    enabled_in_mobile_rasp_tr = models.BooleanField(
        _(u'Запрашивать на мобильных из ПМ (tr)'), blank=False, null=False, default=False
    )

    t_type = models.ForeignKey("www.TransportType", verbose_name=_(u'тип транспорта'),
                               default=2, related_name='order_partner')

    _billing_datasource_id_attr = 'billing_datasource_id_%s' % settings.ENVIRONMENT

    query_module_name = models.CharField(_(u"Версия модуля"), max_length=255, default=None,
                                         blank=True, null=True)

    media_storage = mds_s3_media_storage

    logo = PartnerLogoImageField(
        verbose_name=_(u'Основной логотип'),
        storage=media_storage, null=False, blank=True, default=''
    )
    logo_ru = PartnerLogoImageField(
        verbose_name=_(u'Логотип русской версии'),
        storage=media_storage, null=False, blank=True, default=''
    )
    logo_ua = PartnerLogoImageField(
        verbose_name=_(u'Логотип украинской версии'),
        storage=media_storage, null=False, blank=True, default=''
    )
    logo_tr = PartnerLogoImageField(
        verbose_name=_(u'Логотип турецкой версии'),
        storage=media_storage, null=False, blank=True, default=''
    )

    logo_svg_ru = PartnerLogoSvgImageField(
        verbose_name=_(u'Цветной логотип русской версии в формате svg'),
        storage=media_storage, null=False, blank=True, default=''
    )
    logo_svg_ua = PartnerLogoSvgImageField(
        verbose_name=_(u'Цветной логотип украинской версии в формате svg'),
        storage=media_storage, null=False, blank=True, default=''
    )
    logo_svg_tr = PartnerLogoSvgImageField(
        verbose_name=_(u'Цветной логотип турецкой версии в формате svg'),
        storage=media_storage, null=False, blank=True, default=''
    )
    logo_svg_com = PartnerLogoSvgImageField(
        verbose_name=_(u'Цветной логотип com версии в формате svg'),
        storage=media_storage, null=False, blank=True, default=''
    )

    logo_svg2png_ru = Svg2PngImageField(
        verbose_name=_(u'Рендеренный svg2png логотип русской версии'),
        upload_to=LogoSvg2PngUploadToGetter('ru'),
        storage=media_storage, null=True, blank=True, default=None
    )

    logo_svg2png_ua = Svg2PngImageField(
        verbose_name=_(u'Рендеренный svg2png логотип украинской версии'),
        upload_to=LogoSvg2PngUploadToGetter('ua'),
        storage=media_storage, null=True, blank=True, default=None
    )

    logo_svg2png_tr = Svg2PngImageField(
        verbose_name=_(u'Рендеренный svg2png логотип турецкой версии'),
        upload_to=LogoSvg2PngUploadToGetter('tr'),
        storage=media_storage, null=True, blank=True, default=None
    )

    logo_svg2png_com = Svg2PngImageField(
        verbose_name=_(u'Рендеренный svg2png логотип com версии'),
        upload_to=LogoSvg2PngUploadToGetter('com'),
        storage=media_storage, null=True, blank=True, default=None
    )

    review_percent = models.IntegerField(_(u'Процент проверок'),
                                         blank=False, null=False, default=10)

    is_aviacompany = models.BooleanField(
        _(u'Авиакомпания'), blank=False, null=False, default=False
    )

    can_fetch_by_daemon = models.BooleanField(
        _(u'Может быть запрошен через демона'), blank=False, null=False, default=True,
        help_text=_(u'Обычно этот чекбокс должен быть включен.'
                    u'Отключайте только для ж/д партнеров, которые запрашиваются не через демон.')
    )

    billing_order_id = models.IntegerField(verbose_name=_(u'Номер заказа в партнерке'),
                                           blank=True, null=True, default=None)

    class Meta:
        app_label = 'order'
        verbose_name = _(u"партнер")
        verbose_name_plural = _(u"партнеры")

    def __unicode__(self):
        return self.code

    @classmethod
    def find_by_billing_datasource_id(cls, id_):
        return cls.objects.get(**{cls._billing_datasource_id_attr: id_})

    @property
    def billing_datasource_id(self):
        return getattr(self, self._billing_datasource_id_attr)

    def set_billing_datasource_id(self, value):
        setattr(self, self._billing_datasource_id_attr, value)

    def get_national_price(self, national_version):
        cache_field = 'clk_price_obj_%s' % national_version

        if not hasattr(self, cache_field):
            setattr(
                self, cache_field,
                ClickPrice(self, national_version)
            )

        return getattr(self, cache_field)

    @property
    def price(self):
        if not hasattr(self, 'clk_price_obj'):
            self.clk_price_obj = ClickPrice(self)

        return self.clk_price_obj

    @property
    def enabled(self):
        if not settings.ORDER_BILLING_ENABLED:
            return True

        if self.code in ['dohop']:
            return True

        if self.disabled:
            return False

        if self.current_balance > 0:
            return True

        # Не работает: refresh_partners_balance выставил disabled = True
        if self.code in ['agent', 'ufs', 'tickets_ua']:
            return True

        return False

    def get_national_click_price(self, national_version):
        field_name = 'click_price_{}'.format(national_version)

        return getattr(self, field_name)

    def get_national_logo(self, national_version=None):
        if not national_version:
            national_version = get_request().NATIONAL_VERSION

        field_name = 'logo_{}'.format(
            national_version
        )

        national_logo = getattr(self, field_name)

        logo = national_logo or self.logo

        return logo.url if logo else None

    def get_national_logo_svg(self, national_version=None):
        if not national_version:
            national_version = get_request().NATIONAL_VERSION

        field_name = 'logo_svg_{}'.format(
            national_version
        )

        logo = getattr(self, field_name, None)

        return logo.url if logo else None

    def get_national_logo_svg2png(self, national_version):
        field_name = 'logo_svg2png_{}'.format(national_version)
        logo = getattr(self, field_name, None)

        return logo.url if logo else None

    def get_staticpages_partner(self, national_version, lang, t_type_code, services):
        try:
            return self.staticpages_partners.get(
                national_version=national_version,
                t_type__code=t_type_code,
                services__in=services
            )

        except (MultipleObjectsReturned, ObjectDoesNotExist):
            return None

    def enabled_in(self, service, national_version, mobile=False):
        if mobile:
            return getattr(self, 'enabled_in_mobile_%s_%s' % (service, national_version), False)

        return getattr(self, 'enabled_in_%s_%s' % (service, national_version), False)

    @property
    def L_national_title(self):
        field_name = 'L_national_{}_title'.format(
            get_request().NATIONAL_VERSION
        )

        return getattr(self, field_name, self.L_title)

    def __json__(self, extended=False):
        return {
            'title': self.L_national_title() or self.title,
            'logo': self.get_national_logo(),
        }

    @classmethod
    def get_actual(cls, t_type_code, national_version='ru', from_rasp=False, mobile=False):
        partners = Partner.objects.filter(t_type__code=t_type_code)

        service = 'rasp' if from_rasp else 'ticket'

        return [
            p for p in partners
            if p.enabled and p.enabled_in(service, national_version, mobile)
        ]

    @classmethod
    @cache_until_switch_thread_safe
    def get_info(cls, code):
        partner = cls.objects.get(code=code)
        return partner.title, partner.site_url

    @classmethod
    def get_train_partner_code(cls, point_from, point_to, national_version):
        if not point_from.country or not point_to.country:
            return None

        # RASP-14147
        if national_version == 'ru':
            if point_from.country.code == point_to.country.code == 'UA':
                return 'ukrmintrans_train'

            # RASP-8130: не обращаться к УФС если обе станции вне РФ
            if point_from.country.code == 'RU' or point_to.country.code == 'RU':
                return 'ufs'

        elif national_version == 'ua':
            if point_from.country.code == point_to.country.code == 'RU':
                return 'ufs'

            if point_from.country.code == 'UA' or point_to.country.code == 'UA':
                return 'ukrmintrans_train'

        return None


class ClickPrice(object):
    def __init__(self, partner_object, national_version=None):
        self.partner = partner_object
        self.national_version = national_version

    @property
    def cents(self):
        return int(self.dollars * 100)

    def get_cents_dispersed(self):
        return int(round_dispersed(float(self.dollars) * 100))

    @property
    def dollars(self):
        if self.national_version:
            return self.partner.get_national_click_price(self.national_version)

        return self.partner.click_price

    def __str__(self):
        return self.dollars


class PartnerEmail(models.Model):
    partner = models.ForeignKey(Partner, related_name='emails')
    email = models.EmailField(blank=False)

    class Meta:
        app_label = 'order'
        verbose_name = _(u"Email поставщика")
        verbose_name_plural = _(u"Email'ы поставщика")


class RegionalizePartnerQueryRule(models.Model):
    check_fields = ('settlement', 'country', 'pseudo_region')

    partner = models.ForeignKey(Partner, null=False, blank=False, verbose_name=_(u"партнер"))
    exclude = models.BooleanField(verbose_name=_(u'кроме'), default=False)

    settlement_from = models.ForeignKey('www.Settlement', null=True, blank=True,
                                        related_name=u"regionalize_query_from",
                                        verbose_name=_(u"город отправления"))

    country_from = models.ForeignKey('www.Country', null=True, blank=True,
                                     related_name=u"regionalize_query_from",
                                     verbose_name=_(u"страна отправления"))

    pseudo_region_from = models.ForeignKey('www.PseudoRegion', null=True, blank=True,
                                           related_name=u"regionalize_query_from",
                                           verbose_name=_(u"псевдорегион отправления"))

    settlement_to = models.ForeignKey('www.Settlement', null=True, blank=True,
                                      related_name=u"regionalize_query_to",
                                      verbose_name=_(u"город прибытия"))

    country_to = models.ForeignKey('www.Country', null=True, blank=True,
                                   related_name=u"regionalize_query_to",
                                   verbose_name=_(u"страна прибытия"))

    pseudo_region_to = models.ForeignKey('www.PseudoRegion', null=True, blank=True,
                                         related_name=u"regionalize_query_to",
                                         verbose_name=_(u"псевдорегион прибытия"))

    user_settlement = models.ForeignKey('www.Settlement', null=True, blank=True,
                                        related_name=u"regionalize_query_user",
                                        verbose_name=_(u"город пользователя"))

    user_country = models.ForeignKey('www.Country', null=True, blank=True,
                                     related_name=u"regionalize_query_user",
                                     verbose_name=_(u"страна пользователя"))

    user_pseudo_region = models.ForeignKey('www.PseudoRegion', null=True, blank=True,
                                           related_name=u"regionalize_query_user",
                                           verbose_name=_(u"псевдорегион пользователя"))

    class Meta:
        app_label = 'order'
        verbose_name = _(u"правило регионализации")
        verbose_name_plural = _(u"регионализация запросов к партнерам")

    def __unicode__(self):
        return _(u"Регионализация %s") % self.partner.code

    def get_point_from(self):
        for field_name in self.check_fields:
            if getattr(self, '%s_from_id' % field_name):
                return getattr(self, '%s_from' % field_name)

    def get_point_to(self):
        for field_name in self.check_fields:
            if getattr(self, '%s_to_id' % field_name):
                return getattr(self, '%s_to' % field_name)

    def get_user_point(self):
        for field_name in self.check_fields:
            if getattr(self, 'user_%s_id' % field_name):
                return getattr(self, 'user_%s' % field_name)


class BlablacarToken(models.Model):
    token = models.TextField(verbose_name=_(u'Токен'), null=False, blank=False)

    expire_msk_dt = models.DateTimeField(verbose_name=_(u'Дата окончания действия токена'),
                                         blank=False, null=False)

    receive_msk_dt = models.DateTimeField(verbose_name=_(u'Дата получения токена'),
                                          blank=False, null=False)

    @classmethod
    def get_latest_token_or_none(cls):
        try:
            return cls.objects.latest('expire_msk_dt')

        except cls.DoesNotExist:
            return None

    def __unicode__(self):
        return u'<Token: %s %s>' % (self.expire_msk_dt, self.token[:8])

    class Meta:
        app_label = 'order'
        verbose_name = _(u'Токен для блаблакар')
        verbose_name_plural = _(u'Токены для блаблакар')


@deconstructible
class DohopVendorLogoSvgUploadToGetter(object):
    def __init__(self, lang):
        self.lang = lang

    def __call__(self, instance, filename):
        return 'data/partners/dohop_svg/dohop_{}_{}.svg'.format(instance.id, self.lang)


@deconstructible
class DohopVendorLogoSvg2PngUploadToGetter(object):
    def __init__(self, lang):
        self.lang = lang

    def __call__(self, instance, filename):
        return 'data/partners/dohop_svg2png/id{}_{}.png'.format(instance.id, self.lang)


class DohopVendor(models.Model):
    L_title = L_field(_(u'Название партнера'), default=u'', max_length=100, blank=True,
                      add_local_field=True, local_field_critical=True,)

    L_national_ru_title = L_field(
        _(u'Название в русской версии'),
        default=u'', max_length=100, blank=True, base_lang='ru',
        fallback_fields=['L_title']
    )

    L_national_ua_title = L_field(
        _(u'Название в украинской версии'),
        default=u'', max_length=100, blank=True, base_lang='uk',
        fallback_fields=['L_title']
    )

    L_national_tr_title = L_field(
        _(u'Название в турецкой версии'),
        default=u'', max_length=100, blank=True, base_lang='tr',
        fallback_fields=['L_title']
    )

    L_national_com_title = L_field(
        _(u'Название в com версии'),
        default=u'', max_length=100, blank=True, base_lang='en',
        fallback_fields=['L_title']
    )

    L_national_com = L_field(
        _(u'Deprecated'),
        default=u'', max_length=100, blank=True, base_lang='en',
        fallback_fields=['L_title']
    )

    description_ru_ru = models.TextField(
        verbose_name=_(u'Описание в русской версии'), null=True, blank=True
    )

    description_ua_uk = models.TextField(
        verbose_name=_(u'Описание в украинской версии (uk)'), null=True, blank=True
    )

    description_ua_ru = models.TextField(
        verbose_name=_(u'Описание в украинской версии (ru)'), null=True, blank=True
    )

    description_tr_tr = models.TextField(
        verbose_name=_(u'Описание в турецкой версии (tr)'), null=True, blank=True
    )

    description_tr_en = models.TextField(
        verbose_name=_(u'Описание в турецкой версии (en)'), null=True, blank=True
    )

    dohop_code = models.CharField(_(u'Код вендора'), max_length=100, null=False, blank=False)
    dohop_id = models.PositiveIntegerField(_(u'ID вендора'), null=False, blank=False, db_index=True)
    dohop_vendor_info = models.TextField(null=False, blank=False,)
    dohop_residences = models.TextField(null=False, blank=False)
    dohop_langs = models.CharField(max_length=255, null=False, blank=False)

    review_percent = models.IntegerField(_(u'Процент проверок'), blank=False, null=False, default=0)

    media_storage = mds_s3_media_storage

    logo_svg_ru = models.FileField(
        verbose_name=_(u'Цветной логотип русской версии в формате svg'),
        upload_to=DohopVendorLogoSvgUploadToGetter('ru'),
        storage=media_storage, null=True, blank=True, default=None,
    )

    logo_svg_ua = models.FileField(
        verbose_name=_(u'Цветной логотип украинской версии в формате svg'),
        upload_to=DohopVendorLogoSvgUploadToGetter('ua'),
        storage=media_storage, null=True, blank=True, default=None,
    )

    logo_svg_tr = models.FileField(
        verbose_name=_(u'Цветной логотип турецкой версии в формате svg'),
        upload_to=DohopVendorLogoSvgUploadToGetter('tr'),
        storage=media_storage, null=True, blank=True, default=None,
    )

    logo_svg_com = models.FileField(
        verbose_name=_(u'Цветной логотип com версии в формате svg'),
        upload_to=DohopVendorLogoSvgUploadToGetter('com'),
        storage=media_storage, null=True, blank=True, default=None,
    )

    logo_svg2png_ru = models.ImageField(
        verbose_name=_(u'Рендеренный svg2png логотип русской версии'),
        upload_to=DohopVendorLogoSvg2PngUploadToGetter('ru'),
        storage=media_storage, null=True, blank=True, default=None
    )

    logo_svg2png_ua = models.ImageField(
        verbose_name=_(u'Рендеренный svg2png логотип украинской версии'),
        upload_to=DohopVendorLogoSvg2PngUploadToGetter('ua'),
        storage=media_storage, null=True, blank=True, default=None
    )

    logo_svg2png_tr = models.ImageField(
        verbose_name=_(u'Рендеренный svg2png логотип турецкой версии'),
        upload_to=DohopVendorLogoSvg2PngUploadToGetter('tr'),
        storage=media_storage, null=True, blank=True, default=None
    )

    logo_svg2png_com = models.ImageField(
        verbose_name=_(u'Рендеренный svg2png логотип com версии'),
        upload_to=DohopVendorLogoSvg2PngUploadToGetter('com'),
        storage=media_storage, null=True, blank=True, default=None
    )

    enabled_in_ticket_ru = models.BooleanField(_(u'Запрашивать из КБ (ru)'), blank=False, null=False, default=False)
    enabled_in_ticket_ua = models.BooleanField(_(u'Запрашивать из КБ (ua)'), blank=False, null=False, default=False)
    enabled_in_ticket_tr = models.BooleanField(_(u'Запрашивать из КБ (tr)'), blank=False, null=False, default=False)
    enabled_in_ticket_com = models.BooleanField(_(u'Запрашивать из КБ (com)'), blank=False, null=False, default=False)

    enabled_in_rasp_ru = models.BooleanField(_(u'Запрашивать из ПМ (ru)'), blank=False, null=False, default=False)
    enabled_in_rasp_ua = models.BooleanField(_(u'Запрашивать из ПМ (ua)'), blank=False, null=False, default=False)
    enabled_in_rasp_tr = models.BooleanField(_(u'Запрашивать из ПМ (tr)'), blank=False, null=False, default=False)

    enabled_in_mobile_ticket_ru = models.BooleanField(_(u'Запрашивать на мобильных из КБ (ru)'), blank=False,
                                                      null=False, default=False)
    enabled_in_mobile_ticket_ua = models.BooleanField(_(u'Запрашивать на мобильных из КБ (ua)'), blank=False,
                                                      null=False, default=False)
    enabled_in_mobile_ticket_tr = models.BooleanField(_(u'Запрашивать на мобильных из КБ (tr)'), blank=False,
                                                      null=False, default=False)
    enabled_in_mobile_ticket_com = models.BooleanField(_(u'Запрашивать на мобильных из КБ (com)'), blank=False,
                                                       null=False, default=False)

    enabled_in_mobile_rasp_ru = models.BooleanField(_(u'Запрашивать на мобильных из ПМ (ru)'), blank=False, null=False,
                                                    default=False)
    enabled_in_mobile_rasp_ua = models.BooleanField(_(u'Запрашивать на мобильных из ПМ (ua)'), blank=False, null=False,
                                                    default=False)
    enabled_in_mobile_rasp_tr = models.BooleanField(_(u'Запрашивать на мобильных из ПМ (tr)'), blank=False, null=False,
                                                    default=False)

    enabled = models.BooleanField(verbose_name=_(u'Может быть запрошен через демона'), blank=False, null=False,
                                  default=False)
    dohop_cache_ttl = models.PositiveSmallIntegerField(verbose_name=_(u'Время жизни кеша, минут'), blank=False,
                                                       null=False, default=20,
                                                       validators=[MinValueValidator(0), MaxValueValidator(90)])

    foreign_currency = models.BooleanField(
        verbose_name=_(u'Иностранные валюты'),
        help_text=_(u'Импортировать предложения в валютах '
                    u'отличных от валюты национальной версии'),
        blank=False, null=False, default=False
    )

    foreign_currency = models.BooleanField(
        verbose_name=_(u'Иностранные валюты'),
        help_text=_(u'Импортировать предложения в валютах '
                    u'отличных от валюты национальной версии'),
        blank=False, null=False, default=False
    )

    t_type = models.ForeignKey("www.TransportType", verbose_name=_(u'тип транспорта'),
                               default=2, related_name='+')

    query_module_name = models.CharField(_(u"Версия модуля"), max_length=255, default=None,
                                         blank=True, null=True)

    billing_datasource_id_dev = models.IntegerField(verbose_name=_(u'iD партнера в системе биллинга dev'),
                                                    blank=True, null=True, default=None, help_text=_(u"datasource ID"))
    billing_datasource_id_testing = models.IntegerField(verbose_name=_(u'iD партнера в системе биллинга testing'),
                                                        blank=True, null=True, default=None,
                                                        help_text=_(u"datasource ID"))
    billing_datasource_id_production = models.IntegerField(verbose_name=_(u'iD партнера в системе биллинга production'),
                                                           blank=True, null=True, default=None,
                                                           help_text=_(u"datasource ID"))

    # Deprecated
    click_price = models.FloatField(verbose_name=_(u'стоимость клика, у.е.'),
                                    blank=False, null=False, default=0)

    is_aviacompany = models.BooleanField(
        _(u'Авиакомпания'), blank=False, null=False, default=False
    )

    def enabled_in(self, service, national_version, mobile=False):
        if mobile:
            return getattr(self, 'enabled_in_mobile_%s_%s' % (service, national_version), False)

        return getattr(self, 'enabled_in_%s_%s' % (service, national_version), False)

    @cached_property
    def code(self):
        return 'dohop_%s' % self.dohop_id

    @property
    def billing_datasource_id(self):
        return getattr(self, 'billing_datasource_id_%s' % settings.ENVIRONMENT)

    def L_national_title(self, national_version=None):
        if not national_version:
            national_version = get_request().NATIONAL_VERSION

        field_name = 'L_national_{}_title'.format(
            national_version
        )

        return getattr(self, field_name, self.L_title)()

    def get_national_logo_svg(self, national_version):
        field_name = 'logo_svg_{}'.format(
            national_version
        )

        logo = getattr(self, field_name, None)

        return logo.url if logo else None

    def get_national_price(self, national_version):
        cache_field = 'clk_price_obj_%s' % national_version

        if not hasattr(self, cache_field):
            setattr(
                self, cache_field,
                ClickPrice(
                    Partner.objects.get(code='dohop'),
                    national_version
                )
            )

        return getattr(self, cache_field)

    # Заглушка
    def get_staticpages_partner(self, *args, **kwargs):
        return None

    # Заглушка
    def get_national_logo(self, national_version):
        return None

    def __unicode__(self):
        return self.code

    @classmethod
    def get_actual(cls, national_version='ru', from_rasp=False, mobile=False):
        vendors = DohopVendor.objects.filter(enabled=True)

        service = 'rasp' if from_rasp else 'ticket'

        return [
            p for p in vendors
            if p.enabled_in(service, national_version, mobile)
        ]

    class Meta:
        app_label = 'order'
        verbose_name = _(u"вендор Dohop")
        verbose_name_plural = _(u"вендоры Dohop")
