# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from itertools import groupby

from django.db import models
from django.utils.translation import ugettext_lazy as _
from travel.rasp.library.python.common23.date.run_mask import RunMask
from travel.rasp.library.python.common23.models.core.schedule.avia_alliance import AviaAlliance  # noqa
from travel.rasp.library.python.common23.models.core.schedule.base_rthread import BaseRThread  # noqa
from travel.rasp.library.python.common23.models.core.schedule.base_supplier import BaseSupplier  # noqa
from travel.rasp.library.python.common23.models.core.schedule.company import Company  # noqa
from travel.rasp.library.python.common23.models.core.schedule.express_type import EXPRESS_TYPE_CHOICES, ExpressType  # noqa
from travel.rasp.library.python.common23.models.core.schedule.express_type_lite import ExpressTypeLite  # noqa
from travel.rasp.library.python.common23.models.core.schedule.route import Route  # noqa
from travel.rasp.library.python.common23.models.core.schedule.rthread import RThread  # noqa
from travel.rasp.library.python.common23.models.core.schedule.rthread_type import RThreadType  # noqa
from travel.rasp.library.python.common23.models.core.schedule.rtstation import RTStation  # noqa
from travel.rasp.library.python.common23.models.core.schedule.rtstation_logic_mixin import RTStationLogicMixin  # noqa
from travel.rasp.library.python.common23.models.core.schedule.station_schedule import StationSchedule  # noqa
from travel.rasp.library.python.common23.models.core.schedule.supplier import Supplier  # noqa
from travel.rasp.library.python.common23.models.core.schedule.train_schedule_plan import TrainSchedulePlan  # noqa
from travel.rasp.library.python.common23.models.core.schedule.train_purchase_number import TrainPurchaseNumber  # noqa
from travel.rasp.library.python.common23.models.fields import TrimmedCharField, ThreadCalendarField, RegExpField
from travel.rasp.library.python.common23.models.texts.platform_translation import PlatformTranslation  # noqa
from travel.rasp.library.python.common23.models.transport.deluxe_train import DeLuxeTrain  # noqa
from travel.rasp.library.python.common23.models.transport.vehicle_producer import VehicleProducer  # noqa

log = logging.getLogger(__name__)


class CompanyMarker(models.Model):
    """Пометка для ключевых станций перевозчика"""
    company = models.ForeignKey(Company, blank=False, verbose_name=_(u"перевозчик"))
    station = models.ForeignKey('www.Station', blank=False, verbose_name=_(u"станция"))

    class Meta:
        verbose_name = _(u'маркер перевозчика')
        verbose_name_plural = _(u'маркеры перевозчиков')
        unique_together = 'company', 'station'
        app_label = 'www'
        db_table = 'www_companymarker'


class PlatformRepresentation(models.Model):
    """Регулярки для представления платформ ДЖВ в правильном виде"""
    station = models.ForeignKey('www.Station', blank=False, verbose_name=_('станция'))
    reg_exp = RegExpField(max_length=255, verbose_name=_('регулярное выражение'))
    representation = models.CharField(max_length=255, verbose_name=_('представление'))

    class Meta:
        verbose_name = _('представление платформы')
        verbose_name_plural = _('представления платформ')
        app_label = 'www'
        db_table = 'www_platformpresentation'


class ScheduleExclusion(models.Model):
    """
    Исключения из табло и расписаний
    """
    number = TrimmedCharField(verbose_name=_(u"Номер рейса"), max_length=255, null=True, blank=True,
                              help_text=_(u"Если номера нет, берутся все рейсы между станциями"))
    start_station = models.ForeignKey('www.Station', verbose_name=_(u"Начальная станция маршрута"),
                                      null=False, blank=False, related_name='scheduleexclusion_start_set')
    end_station = models.ForeignKey('www.Station', verbose_name=_(u"Конечная станция маршрута"),
                                    null=False, blank=False, related_name='scheduleexclusion_end_set')
    exclude_station = models.ForeignKey('www.Station', verbose_name=_(u"Исключаемая стания"),
                                        null=False, blank=False)

    departure = models.TimeField(_(u"Оптправление с начальной станции"), null=True, blank=True)
    departure_interval = models.TimeField(_(u"Конец интервала отправления"), null=True, blank=True,
                                          default=None)

    def get_thread_station_pairs(self):
        from django.db import connection

        cursor = connection.cursor()
        cursor.execute("""
        SELECT DISTINCT t.id FROM www_rthread t
            JOIN www_rtstation rts_start ON rts_start.thread_id = t.id AND rts_start.tz_arrival IS NULL
            JOIN www_rtstation rts_end ON rts_end.thread_id = t.id AND rts_end.tz_departure IS NULL
        WHERE rts_start.station_id = %s AND rts_end.station_id = %s
        """, [self.start_station.id, self.end_station.id])

        ids = [row[0] for row in cursor.fetchall()]

        threads = RThread.objects.filter(pk__in=ids)

        if self.number:
            threads = threads.filter(number=self.number)

        if self.departure is not None and self.departure_interval is not None:
            threads = threads.filter(tz_start_time__range=(self.departure, self.departure_interval))
        elif self.departure is not None:
            threads = threads.filter(tz_start_time=self.departure)

        pairs = []

        for t in threads:
            pairs.append((t, self.exclude_station))

        return pairs

    @classmethod
    def get_all_exclusion_pairs(cls):
        pairs = []
        for exclusion in cls.objects.all():
            pairs += exclusion.get_thread_station_pairs()

        return pairs

    @classmethod
    def get_excluded_thread_for_station(cls, station):
        threads = []

        for exclusion in cls.objects.filter(exclude_station=station):
            for pair in exclusion.get_thread_station_pairs():
                threads.append(pair[0])

        return threads

    class Meta:
        verbose_name = _(u'Исключение из расписания по станции')
        verbose_name_plural = _(u'Исключения из расписания по станции')
        app_label = 'www'
        db_table = 'importinfo_scheduleexclusion'


class TrainTurnover(models.Model):
    """
    Информация об обороте поездов.
    https://st.yandex-team.ru/RASPEXPORT-259
    """
    station = models.ForeignKey('www.Station', verbose_name=_('Cтанция оборота'),
                                null=True, blank=True)
    start_date = models.DateField(verbose_name=_('Начало хождения шаблона'), null=True,
                                  blank=True)
    end_date = models.DateField(verbose_name=_('Конец хождения шаблона'), null=True,
                                blank=True)
    template_code = models.CharField(verbose_name=_('Код шаблона дней хождений'),
                                     max_length=100, default=None,
                                     blank=True, null=True)
    year_days = ThreadCalendarField(verbose_name=_('Дни хождений'), default=RunMask.EMPTY_YEAR_DAYS)
    graph = models.ForeignKey('TrainSchedulePlan', verbose_name=_('График поездов'),
                              default=None, blank=True, null=True)
    number_before = models.CharField(verbose_name=_('Номер'), max_length=100, db_index=True,
                                     null=False, blank=True, default='')
    number_after = models.CharField(verbose_name=_('Номер'), max_length=100, db_index=True,
                                    null=False, blank=True, default='')

    class Meta:
        verbose_name = _('Оборот поезда')
        verbose_name_plural = _('Обороты поездов')
        app_label = 'www'
        unique_together = (('number_before', 'station', 'template_code', 'graph'),)
