# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import six
from common.utils.regexp import RegExpSet

from django.core import validators
from django.db import models
from django.utils.encoding import smart_text
from django.utils.translation import ugettext_lazy as _
from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement
from travel.rasp.library.python.common23.models.core.geo.station import Station
from travel.rasp.library.python.common23.models.fields import CodeCharField, RegExpField, TrimmedCharField
from travel.rasp.library.python.common23.models.precache.manager import PrecachingManager
from travel.rasp.library.python.common23.models.tariffs.aeroex_tariff import AeroexTariff  # noqa
from travel.rasp.library.python.common23.models.tariffs.base_thread_tariff import BaseThreadTariff  # noqa
from travel.rasp.library.python.common23.models.tariffs.tariff_group import TariffGroup  # noqa
from travel.rasp.library.python.common23.models.tariffs.tariff_type import TariffType  # noqa
from travel.rasp.library.python.common23.models.tariffs.thread_tariff import ThreadTariff  # noqa
from travel.rasp.library.python.common23.xgettext.i18n import mark_gettext

log = logging.getLogger(__name__)


# Имена классов для отображения, в порядке отображения
# Закомментированные отображать не нужно
CLS_MAPPING = [
    # классы тарифов для ж/д билетов
    (mark_gettext('СВ'), 'suite'),
    (mark_gettext('купе'), 'compartment'),
    (mark_gettext('плацкарт'), 'platzkarte'),
    (mark_gettext('сидячие'), 'sitting'),
    (mark_gettext('общий'), 'common'),
    # (mark_gettext(u'М'), 'soft'),

    # классы тарифов для авиа билетов
    (mark_gettext('бизнес'), 'business'),
    (mark_gettext('эконом'), 'economy'),

    # фейковые классы тарифов без отображаемых названий
    ('', 'bus'),
    ('', 'blablacar'),
]


ORDER_CLS_MAPPING = [
    ('common', mark_gettext('общий')),
    ('sitting', mark_gettext('сидячие')),
    ('platzkarte', mark_gettext('плацкарт')),
    ('compartment', mark_gettext('купе')),
    ('compartment_min', mark_gettext('купе')),
    ('compartment_max', mark_gettext('купе эконом-класса')),
    ('suite', mark_gettext('люкс (СВ)')),
    ('soft', mark_gettext('мягкие')),

    # Самолеты
    ('economy', mark_gettext('эконом')),
    ('business', mark_gettext('бизнес')),
    ('first', mark_gettext('первый')),
]

CLS_NAMES = dict(ORDER_CLS_MAPPING)


@six.python_2_unicode_compatible
class TCHDirection(models.Model):
    iata_from = CodeCharField(verbose_name=_('IATA код отправления'),
                              max_length=10, null=False)

    iata_to = CodeCharField(verbose_name=_('IATA код прибытия'),
                            max_length=10, null=False)

    def __str__(self):
        title_from = get_point_title_by_iata(self.iata_from)
        title_to = get_point_title_by_iata(self.iata_to)
        return 'TCH направление %s %s, "%s" "%s"' % (
            title_from, title_to, self.iata_from, self.iata_to)

    class Meta:
        app_label = 'tariffs'
        verbose_name = _('направление обслуживаемое TCH')
        verbose_name_plural = _('направления для TCH')
        unique_together = (('iata_from', 'iata_to'),)


def get_point_title_by_iata(iata):
    try:
        point = Station.get_by_code('iata', iata)
    except Station.DoesNotExist:
        point = None
    if point is None:
        try:
            point = Settlement.objects.get(iata=iata)
        except Settlement.DoesNotExist:
            point = None

    return point.title if point else iata


@six.python_2_unicode_compatible
class SuburbanTariff(models.Model):
    code = TrimmedCharField(verbose_name=_('код'), max_length=10, primary_key=True)
    tariff = models.FloatField(verbose_name=_('цена'))

    objects = PrecachingManager(keys=['pk', 'code'])

    @classmethod
    def get_tariff(cls, code):
        try:
            return cls.objects.get(code=code).tariff
        except SuburbanTariff.DoesNotExist:

            log.error('Tariff with code %s not found', code)

            return None

    class Meta:
        verbose_name = _('тариф пригородного транспорта')
        verbose_name_plural = _('тарифы пригородного транспорта')
        app_label = 'www'

    def __str__(self):
        return self.code


class NotTariffTrain(models.Model):
    u"""Поезда, которые не тарифицирует Экспресс

    Номер это регулярное выражение
    """
    number = RegExpField(verbose_name=_('номер рейса'), max_length=255, unique=True,
                         blank=False, help_text=_('номер рейса можно задавать регулярым выражением'))

    class Meta:
        verbose_name = _('поезд, который не тарифицирует Экспресс')
        verbose_name_plural = _('поезда, которые не тарифицирует Экспресс')
        app_label = 'www'

    _number_set = []

    @classmethod
    def precache_numbers(cls):
        cls._number_set = RegExpSet([ntt.number for ntt in cls.objects.all()])

    @classmethod
    def number_set(cls, not_cache=False):
        if not_cache:
            return RegExpSet([ntt.number for ntt in cls.objects.all()])

        return cls._number_set


class SuburbanTariffProvider(object):
    AEROEXPRESS = 'aeroexpress'
    MOVISTA = 'movista'
    IM = 'im'

    ALL = {AEROEXPRESS, MOVISTA, IM}
    PRIORITY = [MOVISTA, AEROEXPRESS, IM]


class SuburbanSellingFlow(object):
    AEROEXPRESS = 'aeroexpress'
    VALIDATOR = 'validator'
    SIMPLE = 'simple'

    ALL = {AEROEXPRESS, VALIDATOR, SIMPLE}


class SuburbanSellingBarcodePreset(object):
    PDF417_CPPK = 'PDF417_cppk'
    PDF417_SZPPK = 'PDF417_szppk'
    AZTEC_MTPPK = 'Aztec_mtppk'
    NO_BARCODE = 'No_barcode'


class TariffTypeCode(object):
    USUAL = 'etrain'
    EXPRESS = 'express'
    AEROEXPRESS = 'aeroexpress'

    @classmethod
    def values(cls):
        return [cls.USUAL, cls.EXPRESS, cls.AEROEXPRESS]


SETTINGS_TYPES = {
    'str': (_('Строка'), six.text_type),
    'int': (_('Целое'), int),
    'float': (_('Число с плавающей точкой'), float),
}


class Setting(models.Model):
    name = TrimmedCharField(_('Название'), max_length=255, blank=False, null=False)
    code = CodeCharField(
        _('Код'), max_length=255, blank=False, null=False,
        validators=[
            validators.RegexValidator(
                '(?u)^[A-Z][A-Z_0-9]*$',
                _('Код должен начинаться с буквы, допустимые символы [A-Z_0-9]')
            )
        ]
    )
    type = models.CharField(
        _('Тип'), max_length=20,
        choices=list(  # list(...) не засоряет неймспейс в отличие от [...]
            (type_code, type_text)
            for type_code, (type_text, _type) in SETTINGS_TYPES.items()
        ),
        blank=False, null=False, default='str'
    )
    value = models.CharField(_('Значение'), max_length=1024,
                             blank=True, null=False, default="")
    help_text = models.TextField(_('Подсказка'), max_length=1024,
                                 blank=True, null=True, default="")

    def get_value(self):
        return SETTINGS_TYPES[self.type][1](self.value)

    def clean(self):
        try:
            self.get_value()
        except Exception as e:
            raise validators.ValidationError(
                _('Не удалось преобразовать %s к типу %s: %s') %
                (self.value, self.get_type_display(), smart_text(e)))

    objects = PrecachingManager(keys=['code'])

    @classmethod
    def get(cls, code):
        return cls.objects.get(code=code).get_value()

    @classmethod
    def get_settings_as_class(cls):
        class S:
            pass

        s = S()

        for code, value in cls.get_settings_as_dict().items():
            setattr(s, code, value)

        return s

    @classmethod
    def get_settings_as_dict(cls):
        d = {}

        for s in cls.objects.all():
            d[s.code] = s.get_value()

        return d

    class Meta:
        app_label = 'www'
        verbose_name = _('Настройка')
        verbose_name_plural = _('Настройки')
