# coding: utf-8

from __future__ import absolute_import

from itertools import groupby

from django.db import models
from django.db.models import Q
from django.utils.translation import ugettext_lazy as _

from common.models.geo import Region, Station

class RegionWhiteList(models.Model):
    regions = models.ManyToManyField(Region, blank=True, verbose_name=_(u'области'))

    class Meta:
        app_label = 'www'
        verbose_name = _(u'белый список областей для экспорта')
        verbose_name_plural = _(u'белый список областей для экспорта')


class RoutePath(models.Model):
    """
    Геокодированный сегмент маршрута.

    Задача: для маршрутов необходимо на карте рисовать путь по "дорогам" с учетом фактического движения транспорта.

    Требования:
        1) Несколько маршрутов могут иметь одинаковые участки. Контент менеджер не должен дублировать работу по
           геокодированию одинаковых участков.

    Соглашения:
        1) Для всех маршрутов между станциями A и B геометрия (маршрут следования) одинаковая
        2) Для части сегментов надо уметь задавать различные геометрии для направлений туда и обратно

    Реализация:
        1) Маршрут от станции "А" до станции "Г" разбивается на сегменты по промежуточным станциям "А"-"Б", "Б"-"В",
           "В"-"Г". Каждый сегмент маршрута контент-менеджер геокодирует.
        2) Направлением "прямым" считается направление от станции с меньшим id к станции с большим id, а
           "обратным" от станции с большим id к станции с меньшим id, а
    """

    STATUS_REMOVED = 0
    STATUS_CONFIRMED = 1
    STATUS_CHANGED = 2

    station_from = models.ForeignKey(Station, verbose_name=_(u'от станции'),
                                     help_text=_(u'начало геокодированного участка'),
                                     related_name='routesegment_directions_from', blank=True, null=True)

    station_to = models.ForeignKey(Station, verbose_name=_(u'до станции'),
                                   help_text=_(u'конец геокодированного участка'),
                                   related_name='routesegment_directions_to', blank=True, null=True)

    for_two_directions = models.BooleanField(_(u'для двух направлений'))

    data_direct = models.TextField(verbose_name=_(u'данные прямого направления'),
                                   help_text=_(u'json-данные геокодирования маршрута'),
                                   null=True, blank=True)

    data_back = models.TextField(verbose_name=_(u'данные обратного направления'),
                                 help_text=_(u'json-данные геокодирования маршрута'),
                                 null=True, blank=True)

    status_direct = models.IntegerField(
        verbose_name=_(u'статус прямого направления'),
        choices=[
            (STATUS_REMOVED, _(u'не геокодировано')),
            (STATUS_CONFIRMED, _(u'геокодировано')),
            (STATUS_CHANGED, _(u'станции передвинуты'))
        ],
        default=STATUS_REMOVED
    )

    status_back = models.IntegerField(
        verbose_name=_(u'статус обратного направления'),
        choices=[
            (STATUS_REMOVED, _(u'не геокодировано')),
            (STATUS_CONFIRMED, _(u'геокодировано')),
            (STATUS_CHANGED, _(u'станции передвинуты'))
        ],
        default=STATUS_REMOVED)

    class Meta:
        verbose_name = _(u'геокодированный участок маршрута')
        verbose_name_plural = _(u'геокодированные участки маршрута')
        app_label = 'www'

    @classmethod
    def get(cls, station_from, station_to):
        """
        Ищет участок маршрута между станциями station_from и station_to
        """

        station_from, station_to, direction = cls._order_stations(station_from, station_to)

        return cls._wrap(cls._get(station_from, station_to), direction, station_from, station_to)

    @classmethod
    def get_route_paths_index(cls, segments):
        stations_from = []
        stations_to = []
        path_keys = set()

        for station_from, station_to in segments:
            station_from, station_to, _ = cls._order_stations(station_from, station_to)
            stations_from.append(station_from)
            stations_to.append(station_to)
            path_keys.add(cls._get_path_key(station_from, station_to))

        route_paths = cls.objects.filter(Q(station_from_id__in=stations_from, station_to_id__in=stations_to))\
            .select_related('station_from', 'station_to')

        route_paths = [rp for rp in route_paths if cls._get_path_key(rp.station_from, rp.station_to) in path_keys]

        return {(rp.station_from, rp.station_to): rp for rp in route_paths}

    @classmethod
    def get_route_paths(cls, stations):
        """
        Строит список геокодированных участков маршрута между станциями stations[0] и stations[-1] с учетом
        промежуточных станции stations[1, -1]
        """

        if len(stations) < 2:
            return []

        segments = [(stations[i], stations[i + 1]) for i in xrange(len(stations) - 1)]
        route_paths_index = cls.get_route_paths_index(segments)
        result = []

        for i in xrange(len(stations) - 1):
            station_from, station_to, direction = cls._order_stations(stations[i], stations[i + 1])

            route_path = route_paths_index.get((station_from, station_to), None)

            result.append(cls._wrap(route_path, direction, station_from, station_to))

        return result

    @classmethod
    def bulk_load(cls, segments):
        if not segments:
            return []

        route_paths_index = cls.get_route_paths_index(segments)
        result = []

        for station_from, station_to in segments:
            station_from, station_to, direction = cls._order_stations(station_from, station_to)

            route_path = route_paths_index.get((station_from, station_to), None)

            result.append(cls._wrap(route_path, direction, station_from, station_to))

        return result

    @classmethod
    def is_route_paths_confirmed(cls, route_stations):
        """
        Проверяет множество маршрутов на то, что они полоностью геокодированы. Оптимизировано по производительности
        для большого кол-ва маршрутов

        route_stations - список пар (route_id, station_id) отсортированный по route_id и порядку следования станции в
        маршруте

        Возвращает словарь {route_id, Boolean - полностью геокодирован или нет маршрут}
        """

        route_id_position = 0
        station_id_position = 1

        segments = set()

        for route_id, values in groupby(route_stations, lambda r: r[route_id_position]):
            values = list(values)

            for i in xrange(len(values) - 1):
                segments.add((values[i][station_id_position], values[i + 1][station_id_position]))

        if not segments:
            return {}

        stations_from = []
        stations_to = []
        path_keys = set()

        for s in segments:
            station_from, station_to, _ = cls._order_stations(s[0], s[1])
            stations_from.append(station_from)
            stations_to.append(station_to)
            path_keys.add(cls._get_path_key(station_from, station_to))

        route_paths = cls.objects.filter(Q(station_from_id__in=stations_from, station_to_id__in=stations_to)) \
            .values('station_from_id', 'station_to_id', 'for_two_directions', 'status_direct', 'status_back')

        route_paths = [rp for rp in route_paths if
                       cls._get_path_key(rp['station_from_id'], rp['station_to_id']) in path_keys]
        route_paths_index = dict(((rp['station_from_id'], rp['station_to_id']), rp) for rp in route_paths)

        result = {}

        for route_id, values in groupby(route_stations, lambda r: r[route_id_position]):
            values = list(values)

            for i in xrange(len(values) - 1):
                station_from, station_to, direction = cls._order_stations(values[i][station_id_position],
                                                                          values[i + 1][station_id_position])

                route_path = route_paths_index.get((station_from, station_to), None)

                if not route_path:
                    result[route_id] = False

                    break

                status_field = 'status_direct' if direction or route_path['for_two_directions'] else 'status_back'

                if route_path[status_field] != cls.STATUS_CONFIRMED:
                    result[route_id] = False

                    break

            if route_id not in result:
                result[route_id] = True

        return result

    @classmethod
    def set_changed(cls, station):
        """
        Помечает все участки маршрутов через станцию station для повторного геокодирования. Это необходимо при изменении
        положения станции
        """

        for route_path in cls.objects.filter(Q(station_from=station) | Q(station_to=station)):
            route_path.status_direct = (cls.STATUS_CHANGED
                                        if route_path.status_direct == cls.STATUS_CONFIRMED
                                        else route_path.status_direct)
            route_path.status_back = (cls.STATUS_CHANGED
                                      if route_path.status_back == cls.STATUS_CONFIRMED
                                      else route_path.status_back)
            route_path.save()

    @classmethod
    def save_route_path(cls, station_from, station_to, data, for_two_directions=True):
        station_from, station_to, direction = cls._order_stations(station_from, station_to)

        route_path = cls._get(station_from, station_to)

        if not route_path:
            route_path = cls()

            route_path.station_from = station_from

            route_path.station_to = station_to

        route_path.for_two_directions = for_two_directions

        if for_two_directions or direction:
            route_path.data_direct = data

            route_path.status_direct = cls.STATUS_CONFIRMED

        else:
            route_path.data_back = data

            route_path.status_back = cls.STATUS_CONFIRMED

        route_path.save()

    @classmethod
    def remove_route_path(cls, station_from, station_to):
        station_from, station_to, direction = cls._order_stations(station_from, station_to)

        route_path = cls._get(station_from, station_to)

        if not route_path:
            return

        if direction:
            route_path.data_direct = None

            route_path.status_direct = cls.STATUS_REMOVED

        else:
            route_path.data_back = None

            route_path.status_back = cls.STATUS_REMOVED

        if route_path.status_direct == cls.STATUS_REMOVED and route_path.status_back == cls.STATUS_REMOVED:
            route_path.delete()

        else:
            route_path.save()

    @classmethod
    def _get(cls, station_from, station_to):
        try:
            return cls.objects.get(station_from=station_from, station_to=station_to)

        except cls.DoesNotExist:
            return None

    @classmethod
    def _order_stations(cls, station_from, station_to):
        if cls._get_id(station_from) < cls._get_id(station_to):
            return station_from, station_to, True

        return station_to, station_from, False

    @classmethod
    def _get_id(cls, station):
        return getattr(station, 'id', station)

    @classmethod
    def _get_path_key(cls, station_from, station_to):
        return '-'.join([str(cls._get_id(station_from)), str(cls._get_id(station_to))])

    @classmethod
    def _wrap(cls, route_path, direction, station_from=None, station_to=None):
        if not route_path and not station_to:
            return None

        if not station_from:
            station_from = route_path.station_from

        if not station_to:
            station_to = route_path.station_to

        class Wrapper:
            def __init__(self):
                self.station_from = station_from if direction else station_to
                self.station_to = station_to if direction else station_from
                self.for_two_directions = route_path.for_two_directions if route_path else True

                if route_path:
                    self.status = (route_path.status_direct
                                   if route_path.for_two_directions or direction
                                   else route_path.status_back)
                    self.data = (route_path.data_direct
                                 if route_path.for_two_directions or direction
                                 else route_path.data_back)
                else:
                    self.status = cls.STATUS_REMOVED
                    self.data = None

        return Wrapper()


class RoutePathOwner(object):
    """
    Объект (нитка) имеющий геокодированный путь.

    Такой объект должен реализовывать метод get_stations возвращающий список станций нитки в порядке следования.
    """

    def get_stations(self):
        raise RuntimeError("Method 'get_stations' isn't implemented")

    def is_route_path_confirmed(self):
        """
        Проверяет, что все сегменты геокодированы
        """

        for route_path in self.get_route_paths():
            if not route_path or RoutePath.STATUS_CONFIRMED != route_path.status:
                return False

        return True

    def get_route_paths(self):
        """
        Возвращает список геокодированных сегментов нитки
        """

        return RoutePath.get_route_paths(self.get_stations())
