# coding: utf-8
from __future__ import unicode_literals

import re
from datetime import datetime

import requests
import six
from lxml import html

from common.data_api.file_wrapper.registry import SandboxResources
from common.data_api.sandbox.sandbox_public_api import sandbox_public_api
from common.settings.configuration import Configuration


LISTING_TIMEOUT = 10

DUMP_FILENAME_PATTERNS = [
    re.compile(
        r'^dump_.*_(?P<db_type>work_db|service_db)_(?P<timestamp>\d{14})\.sql\.gz$',
    ),
    re.compile(
        r'^dump_(?P<timestamp>\d{14})_.*_(?P<db_type>work_db|service_db)\.sql\.gz$',
    )
]

SCHEMA_FILENAME_PATTERNS = [
    re.compile(
        r'^schema_(?P<timestamp>\d{14})_.*_(?P<db_type>work_db|service_db)\.sql\.gz$',
    )
]


def parse_filename(filename, patterns):
    for pattern in patterns:
        match = pattern.match(filename)
        if match:
            break
    else:
        return

    return (
        match.group('db_type'),
        datetime.strptime(match.group('timestamp'), '%Y%m%d%H%M%S')
    )


@six.python_2_unicode_compatible
class RemoteDBFile(object):
    def __init__(self, filename, db_type, timestamp, url):
        self.filename = filename
        self.db_type = db_type
        self.timestamp = timestamp
        self.url = url

    def __str__(self):
        return '<RemoteDBFile({filename}, {db_type}, {timestamp}, {url})>'.format(**vars(self))

    def __repr__(self):
        return six.text_type(self)


def get_remote_db_files(base_url, patterns):
    listing_html = requests.get(base_url, timeout=LISTING_TIMEOUT, verify=False).text
    doc = html.document_fromstring(listing_html)
    file_link_els = doc.findall('.//a')

    remote_db_files = []
    for link_el in file_link_els:
        filename = link_el.get('href')
        parsed = parse_filename(filename, patterns)
        if parsed:
            db_type, timestamp = parsed
            remote_db_files.append(RemoteDBFile(filename, db_type, timestamp=timestamp, url=base_url + filename))

    return remote_db_files


def get_dump_remote_files(base_url):
    return get_remote_db_files(base_url, DUMP_FILENAME_PATTERNS)


def get_schema_remote_files(base_url):
    return get_remote_db_files(base_url, SCHEMA_FILENAME_PATTERNS)


def get_latest_dump_url(base_url, db_type=None):
    remote_files = get_dump_remote_files(base_url)  # type: list[RemoteDBFile]

    try:
        return sorted((rf for rf in remote_files if db_type is None or rf.db_type == db_type),
                      key=lambda rf: rf.timestamp, reverse=True)[0].url
    except IndexError:
        return


def get_latest_dump_sandbox_url(db_type=None):
    attrs = {
        'environment': Configuration.PRODUCTION
    }
    if db_type:
        attrs['db_type'] = db_type
    resource = sandbox_public_api.get_resource(resource_type=SandboxResources.MYSQL_DUMP, attrs=attrs, latest=True)

    return resource['http']['proxy']


def get_latest_schema_url(base_url, db_type=None):
    remote_files = get_schema_remote_files(base_url)  # type: list[RemoteDBFile]

    try:
        return sorted((rf for rf in remote_files if db_type is None or rf.db_type == db_type),
                      key=lambda rf: rf.timestamp, reverse=True)[0].url
    except IndexError:
        return
