# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import re
import socket
import time
from xml.parsers.expat import ExpatError

from django.conf import settings
from six.moves import http_client
from six.moves.urllib.parse import urlencode
from six.moves.urllib.request import Request

from common.utils import request_helper
from common.utils.http import quick_urlopen


log = logging.getLogger(__name__)


class BannerError(ValueError):
    pass


def get_banners(page_id, request, params=None, yadirect=False):
    try:
        url = settings.BANNER_URL % page_id

        # Копируем параметры, чтобы не изменять их
        params = [] if params is None else params[:]

        if not yadirect:
            params.append(('direct-limit', 0))

        if hasattr(request, 'wizard_mark'):
            stat_id = {
                'aviawizard': 1,
                'aviawizard1': 2,
            }.get(request.wizard_mark, None)

            if stat_id:
                params.append(('stat-id', stat_id))

        if params:
            url += '?' + urlencode(params)

        log.debug(url)

        banner_request = Request(
            url,
            headers={'X-Real-IP': request.META['REMOTE_ADDR'],
                     'X-Forwarded-For': request.META.get('HTTP_X_FORWARDED_FOR', ''),
                     'User-Agent': request.META.get('HTTP_USER_AGENT', ''),
                     'Cookie': request.META.get('HTTP_COOKIE', ''),
                     'Referer': request_helper.build_absolute_uri(request),
                     }
        )

        start = time.time()
        info = quick_urlopen(banner_request).read().strip().decode('cp1251')
        log.info('1 %s' % (time.time() - start))
        log.debug(info)

        banners = dict((int(p), html) for p, html in re.findall(r'<place(\d+)>(.+?)</place\1>', info, re.S))
        return banners

    except (http_client.HTTPException, IOError, socket.error) as e:
        raise BannerError('Не удалось получить данные из баннерной системы (%s): %s' % (page_id, e))

    except (ExpatError, AttributeError, TypeError) as e:
        raise BannerError('Ошибочный формат ответа баннерной системы (%s): %s' % (page_id, e))

    except Exception:
        raise BannerError('Неизвестная ошибка баннерной системы (%s)' % page_id)
