# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals


import codecs
import os.path
from multiprocessing import Process, Queue

import hemi
import six
from django.conf import settings
from django.utils.encoding import force_text
from django.utils.translation import get_language
from six.moves import filter as ifilter

from common.settings.utils import define_setting
from common.utils.text import NBSP


define_setting('HEMI_RENDER_WORKAROUND', default=False)


class BEMHTML(object):
    def __init__(self, path):
        self.path = path
        self.load()

    def load(self):
        stat = os.stat(self.path)

        with codecs.open(self.path, 'r', encoding='utf-8') as f:
            source = f.read()

        context = hemi.Context()

        context.eval(source)

        self.bemhtml = context.locals.BEMHTML
        self.i18n = context.locals.BEM.I18N

        self.stat = stat

    def apply(self, bemjson, lang=None):
        if settings.DEBUG:
            check_json(bemjson)

            if self.expired():
                self.load()

        if lang is None:
            lang = get_language()

        self.i18n.lang(lang)

        return self.bemhtml.apply(bemjson)

    def expired(self):
        stat = os.stat(self.path)

        for attr in ['st_size', 'st_mtime']:
            if getattr(stat, attr) != getattr(self.stat, attr):
                return True

        return False


def check_json(json):
    if json in [True, False, None, hemi.undefined]:
        return

    if isinstance(json, six.integer_types):
        return

    if isinstance(json, float):
        return

    if isinstance(json, six.binary_type):
        try:
            force_text(json)
        except UnicodeDecodeError as e:
            raise TypeError("hemi won't handle %r: %s" % (json, e))

        return

    if isinstance(json, six.text_type):
        return

    if hasattr(json, '__json__'):
        check_json(json.__json__())

        return

    if isinstance(json, list):
        for item in json:
            check_json(item)

        return

    if isinstance(json, dict):
        for value in json.values():
            check_json(value)

        return

    raise TypeError("hemi won't handle %r" % (json,))


bemhtml_cache = {}


def render(path, bemjson, lang=None):
    def _render(path, bemjson, lang):
        bemhtml = bemhtml_cache.get(path)

        if not bemhtml:
            bemhtml = bemhtml_cache[path] = BEMHTML(path)

        return bemhtml.apply(bemjson, lang=lang)

    if not settings.HEMI_RENDER_WORKAROUND:
        return _render(path, bemjson, lang)

    # Обходим сегфолты запуска под дебаг режимом в pycharm
    def wrap(q, path, bemjson, lang):
        q.put(_render(path, bemjson, lang))

    q = Queue()
    p = Process(target=wrap, args=(q, path, bemjson, lang))
    p.start()
    return q.get()


class Undefined(object):
    def __getattr__(self, name):
        return self

    def __getitem__(self, name):
        return self

    def __iter__(self):
        return iter([])

    def __bool__(self):
        return False

    __nonzero__ = __bool__

    def __json__(self):
        return hemi.undefined

    def __eq__(self, other):
        return other is None


undefined = Undefined()


def wrap(v):
    if isinstance(v, dict):
        return DictWrapper(v)

    if isinstance(v, list):
        return ListWrapper(v)

    return v


class DictWrapper(dict):
    def __getitem__(self, name):
        return wrap(self.get(name, undefined))

    def __getattr__(self, name):
        if name.startswith('__'):
            raise AttributeError

        return self[name]

    def __setattr__(self, name, value):
        self[name] = value


class ListWrapper(list):
    def __getitem__(self, i):
        return wrap(list.__getitem__(self, i))

    def __iter__(self):
        for v in list.__iter__(self):
            yield wrap(v)


def punctuate_content(content, sep=', '):
    punctuated_content = []

    iterator = ifilter(None, content)

    try:
        item = next(iterator)
    except StopIteration:
        return []

    while True:
        punctuated_content.append(item)

        try:
            item = next(iterator)
        except StopIteration:
            break

        punctuated_content.append(sep)

    return punctuated_content


def ru_human_list_ili(l):
    if len(l) < 2:
        return l

    return punctuate_content(l[:-1], sep=', ') + [' или ', l[-1]]


def ru_human_list_i(l):
    if len(l) < 2:
        return l

    return punctuate_content(l[:-1], sep=', ') + [' и' + NBSP, l[-1]]


def tr_human_list(l):
    if len(l) < 2:
        return l

    return punctuate_content(l[:-1], sep=', ') + [' ve ', l[-1]]


class Loop:
    def __init__(self, j, last_index):
        self.index = j
        self.first = not j
        self.last = j == last_index
        self.odd = not j % 2


def loop_over(iterable_list):
    last_index = len(iterable_list) - 1
    return [(Loop(j, last_index), elem)
            for j, elem in enumerate(iterable_list)]


def space(*parts):
    return punctuate_content(parts, sep=' ')
