# -*- coding: utf-8 -*-

import os.path

from django.conf import settings
from django.http import HttpResponse, HttpResponseBadRequest
from django.template.engine import Engine
from django.utils.translation import get_language

from common.utils import bemhtml, request_helper
from common.utils.jsonutils import hemi_json_debug_dumps


class Template(object):
    BEM_TYPE = 'priv.js'

    def headers(self):
        allow_frames = False

        yauser = getattr(self.context.request, 'yauser', None)

        # Включаем фреймы для стаффа яндекса (для вебвизора)
        if yauser and yauser.is_authenticated() and getattr(yauser, 'staff_login', None):
            allow_frames = True

        headers = {}

        if not allow_frames:
            headers['X-Frame-Options'] = 'DENY'

        return headers

    def __init__(self, request, view_context={}):
        context = {
            'request': request,
            'absolute_uri': request_helper.build_absolute_uri(request),
        }

        context.update(view_context)

        default_engine = Engine.get_default()
        for processor in default_engine.template_context_processors:
            context.update(processor(request))

        self.context = bemhtml.wrap(context)

    @classmethod
    def render(cls, *args, **kwargs):
        instance = cls(*args, **kwargs)
        try:
            instance_json = instance.__json__()
        except UnicodeDecodeError:
            return HttpResponseBadRequest("Bad request")

        if settings.DEBUG_BEMJSON and instance.context.request.GET.get('bemjson'):
            bemhtml.check_json(instance_json)

            return HttpResponse(
                hemi_json_debug_dumps(instance_json),
                content_type="application/json; charset=utf-8"
            )

        html = instance.render_bemjson(instance_json)

        response = HttpResponse(html, content_type="text/html")

        for header, value in instance.headers().items():
            response[header] = value

        return response

    def render_bemjson(self, bemjson):
        path = os.path.join(self.PAGE_ROOT, self.page, '_%s.%s' % (self.page, self.BEM_TYPE))

        lang = get_language()

        return bemhtml.render(path, bemjson, lang)


class ChangeParams(object):
    ONE_SHOT_PARAMS = ['noDirect']

    class Positional(object):
        def __init__(self, index, value):
            self.index = index
            self.value = value

        @classmethod
        def get(cls, request_get, name, index):
            try:
                return request_get.getlist(name)[index]
            except IndexError:
                return None

    def change_params(self, **params):
        """
        Ссылка для изменения параметров текущей страницы.

        None удаляет параметр.

        Для генерации ссылок с новыми параметрами используйте urlencode.

        Экранирование амперсанда должно выполняться на уровне bemhtml.
        """

        new_params = self.context.request.GET.copy()

        for param in self.ONE_SHOT_PARAMS:
            if param in new_params:
                del new_params[param]

        for name, value in params.items():
            if value is None:
                if name in new_params:
                    del(new_params[name])

            elif name.endswith('__add'):
                name = name[:-5]

                values = new_params.getlist(name)

                if value not in values:
                    new_params.setlist(name, values + [value])

            elif name.endswith('__remove'):
                name = name[:-8]

                values = new_params.getlist(name)

                if value in values:
                    values.remove(value)

                    new_params.setlist(name, values)

            elif isinstance(value, self.Positional):
                values = new_params.getlist(name)

                # Удлинняем массив, если в него не входит нужный индекс
                if len(values) < value.index + 1:
                    values.extend([''] * (value.index + 1 - len(values)))

                values[value.index] = value.value
                new_params.setlist(name, values)

            elif isinstance(value, list):
                new_params.setlist(name, value)

            elif value is not False:
                new_params[name] = value

        return '?' + new_params.urlencode()
