# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import re

from blackbox import XmlBlackbox, BlackboxError, FIELD_LOGIN
from django.conf import settings
from six.moves.urllib_error import URLError

from common.data_api.tvm.instance import tvm_factory
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from common.utils import request_helper


log = logging.getLogger(__name__)

FIELD_LANG = ('userinfo.lang.uid', 'lang')
FIELD_STAFF_LOGIN = ('subscription.login.669', 'staff_login')

define_setting('BLACKBOX_URL_YATEAM', default='http://blackbox.yandex-team.ru/blackbox')
define_setting('TVM_BLACKBOX_DST', {Configuration.PRODUCTION: 'blackbox-production'}, default='blackbox-testing')


class Attributes(object):
    """ https://wiki.yandex-team.ru/passport/dbmoving/#tipyatributov """
    FIRST_NAME = 27
    LAST_NAME = 28
    GENDER = 29
    BIRTHDAY = 30


def get_session(userip, host, sessionid, ssl_sessionid=None, blackbox_url=None, tvm_ticket=None):

    if not tvm_ticket:
        tvm_provider = tvm_factory.get_provider()
        if tvm_provider:
            tvm_ticket = tvm_provider.get_ticket(settings.TVM_BLACKBOX)
        else:
            log.exception('Не смогли получить tvm_provider')
            return None

    blackbox = XmlBlackbox(
        url=blackbox_url or settings.BLACKBOX_URL,
        dbfields=[FIELD_LOGIN, FIELD_LANG, FIELD_STAFF_LOGIN] + settings.BLACKBOX_PROJECT_FIELDS,
        retry_count=settings.BLACKBOX_RETRY_COUNT,
        timeout=settings.BLACKBOX_TIMEOUT
    )

    sessionid_kwargs = {'headers': {'X-Ya-Service-Ticket': tvm_ticket}}
    if ssl_sessionid is not None:
        sessionid_kwargs['sslsessionid'] = ssl_sessionid

    try:
        session = blackbox.sessionid(sessionid, userip, host, **sessionid_kwargs)
    except (URLError, BlackboxError):
        log.exception(u'Blackbox недоступен.')
        return None

    validate_session(session)
    return session


def validate_session(session):
    if not session:
        return

    if session.status == 'NOAUTH':
        raise SessionNoauth

    if not session.valid:
        raise SessionInvalid

    if session.uid is None:
        raise SessionLite

    if not session.secure:
        raise SessionInsecure


class SessionError(Exception):
    pass


class SessionInvalid(SessionError):
    pass


class SessionNoauth(SessionInvalid):
    pass


class SessionInsecure(SessionError):
    pass


class SessionLite(SessionInsecure):
    pass


_not_provieded = object()


def get_session_params_from_request(request, userip=_not_provieded):
    sessionid = request.COOKIES.get('Session_id')
    ssl_sessionid = request.COOKIES.get('sessionid2')
    host = request_helper.get_host(request)
    if userip is _not_provieded:
        userip = request_helper.get_ip(request)
    return {
        'host': host,
        'sessionid': sessionid,
        'ssl_sessionid': ssl_sessionid,
        'userip': userip
    }


def get_session_for_request(request, userip=_not_provieded, blackbox_url=None, tvm_ticket=None):
    u"""Возвращает валидную защищенную сессию или None при недоступности blackbox"""
    session_params = get_session_params_from_request(request, userip)
    if session_params['sessionid'] is None:
        raise SessionInvalid
    if session_params['ssl_sessionid'] is None:
        raise SessionInsecure

    session = get_session(session_params['userip'], session_params['host'],
                          session_params['sessionid'], session_params['ssl_sessionid'],
                          blackbox_url=blackbox_url, tvm_ticket=tvm_ticket)

    return session


def get_blackbox_oauth_args(request):
    user_ip = request_helper.get_ip(request)
    if user_ip:
        user_ip = user_ip.split(',')[-1]

    oauth_header = request.META.get('HTTP_AUTHORIZATION')
    if not oauth_header:
        raise ValueError('OAuth token must be specified')

    return {
        'oauth_header': oauth_header,
        'user_ip': user_ip,
    }


def get_blackbox_oauth_info(oauth_header, user_ip, blackbox_url=None, tvm_ticket=None, attributes=None, **bb_kw):
    """https://docs.yandex-team.ru/blackbox/methods/oauth"""

    if not (oauth_header and user_ip):
        raise Exception('oauth_header и user_ip должны быть не пустыми')

    if not tvm_ticket:
        tvm_ticket = tvm_factory.get_provider().get_ticket(settings.TVM_BLACKBOX)

    blackbox = XmlBlackbox(
        url=blackbox_url or settings.BLACKBOX_URL,
        retry_count=settings.BLACKBOX_RETRY_COUNT,
        timeout=settings.BLACKBOX_TIMEOUT,
    )

    if attributes:
        bb_kw['attributes'] = ','.join(map(str, attributes))

    try:
        info = blackbox.oauth(
            headers_or_token={
                'Authorization': oauth_header,
                'X-Ya-Service-Ticket': tvm_ticket
            },
            userip=user_ip,
            emails='getdefault',
            **bb_kw
        )
    except Exception:
        log.exception('Ошибка при обращении в Blackbox')
        raise

    if not info['valid']:
        raise SessionInvalid

    # либа не обрезает пробельные символы, которые могут быть в оригинальном xml-ответе blackbox
    emails = info.get('emails')
    if emails:
        for email in emails:
            email['address'] = email['address'].strip()

    return info
