# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import date, timedelta
from itertools import chain
from time import strptime

from lxml import etree
from django.conf import settings

from common.utils.caching import cache_until_switch_thread_safe, cache_until_switch, cache_method_result
from common.utils.date import daterange
from common.utils.fields import MemoryFile
from common.utils.http import urlopen
from common.xgettext.common import xgettext_weekday_short, get_datetemplate_translation
from common.xgettext.i18n import xformat, stringify
from common.importinfo.models import YaCalendarXml


log = logging.getLogger(__name__)


class YCalendarError(Exception):
    pass


class YCalendar(object):
    """Класс для извлечения выходных у Я календаря, по умолчанию Россия"""

    @classmethod
    def get_years(cls, date1, date2):
        years = sorted(list({date1.year, date2.year}))
        # Если больше одного года в диапазоне
        if len(years) == 2:
            years = range(years[0], years[1] + 1)

        return years

    @classmethod
    @cache_until_switch_thread_safe
    def get_holidays(cls, date1, date2, country=None):
        years = cls.get_years(date1, date2)

        all_holidays = []
        for year in years:
            all_holidays += cls.get_holidays_for_year(year, country)

        holidays = set()

        for day in all_holidays:
            if date1 <= day <= date2:
                holidays.add(day)

        return holidays

    @classmethod
    @cache_until_switch_thread_safe
    def get_workdays(cls, date1, date2, country=None):
        years = cls.get_years(date1, date2)

        all_workdays = []
        for year in years:
            all_workdays += cls.get_workdays_for_year(year, country)

        workdays = set()

        for day in all_workdays:
            if date1 <= day <= date2:
                workdays.add(day)

        return workdays

    @classmethod
    @cache_until_switch_thread_safe
    def get_weekends(cls, date1, date2, country=None):
        years = cls.get_years(date1, date2)

        all_weekends = []
        for year in years:
            all_weekends += cls.get_weekends_for_year(year, country)

        weekends = set()

        for day in all_weekends:
            if date1 <= day <= date2:
                weekends.add(day)

        return weekends

    @staticmethod
    @cache_until_switch_thread_safe
    def get_holidays_for_year(year, country):
        u"""
        Только праздники
        """

        ycal_xml = YCalendar.get_ycalendar_xml(year, country)

        holidays = []

        tree = etree.parse(ycal_xml.xml_file)

        for day in tree.getiterator("day"):
            if day.get("day-type") == u"holiday":
                holidays.append(date(*strptime(day.get("date"), "%Y-%m-%d")[:3]))

        return tuple(holidays)

    @staticmethod
    @cache_until_switch_thread_safe
    def get_weekends_for_year(year, country):
        u"""
        Все не рабочие дни
        """

        ycal_xml = YCalendar.get_ycalendar_xml(year, country)

        weekends = []

        tree = etree.parse(ycal_xml.xml_file)

        for day in tree.getiterator("day"):
            if day.get("is-holiday") == u"1":
                weekends.append(date(*strptime(day.get("date"), "%Y-%m-%d")[:3]))

        return tuple(weekends)

    @staticmethod
    @cache_until_switch_thread_safe
    def get_workdays_for_year(year, country):

        ycal_xml = YCalendar.get_ycalendar_xml(year, country)

        workdays = []

        tree = etree.parse(ycal_xml.xml_file)

        for day in tree.getiterator("day"):
            if day.get("is-holiday") == u"0":
                workdays.append(date(*strptime(day.get("date"), "%Y-%m-%d")[:3]))

        return tuple(workdays)

    @staticmethod
    def get_ycalendar_xml(year, country):
        from common.models.geo import Country
        country = country or Country.objects.get(id=Country.RUSSIA_ID)

        try:
            ycal_xml = YaCalendarXml.objects.get(year=year, country=country)
        except YaCalendarXml.DoesNotExist:
            ycal_xml = YCalendar.download_ycalendar_xml(year, country)

        return ycal_xml

    @staticmethod
    def download_ycalendar_xml(year, country):

        date1 = date(year, 1, 1)
        date2 = date(year, 12, 31)

        if not country._geo_id:
            raise YCalendarError(u"Отсутствует _geo_id у страны %s" % country.title)

        url = ("https://calendar.yandex.ru/export/holidays.xml?"
               "start_date=%(start_date)s&end_date=%(end_date)s"
               "&country_id=%(country_geo_id)s&out_mode=all")
        url = url % {
            "start_date": date1,
            "end_date": date2,
            "country_geo_id": country._geo_id
        }

        try_again = True
        tries = 0

        while try_again:
            try:
                tree = etree.parse(urlopen(url))
                try_again = False
            except:
                tries += 1
                if tries > 3:
                    raise

        if not tree.findall('.//day'):
            raise YCalendarError(u"Пустой календарь у страны %s на год %s" % (country.title, year))

        xml_file = MemoryFile(u"calendar_%s_%s_%s.xml" % (year, country._geo_id, country.title),
                              "text/xml", etree.tostring(tree, encoding="utf-8", pretty_print=True,
                                                         xml_declaration=True))

        ycal_xml = YaCalendarXml.objects.create(year=year, country=country, xml_file=xml_file)

        return ycal_xml


class DateTemplate(object):
    has_except = False

    def __init__(self, start_date, length):
        self._start_date = start_date
        self._length = length
        self.days = set(self.get_days())

    def get_text(self, lang):
        return get_datetemplate_translation(lang, type(self).__name__)


class AllDateTemplate(DateTemplate):
    def get_days(self):
        for i in range(self._length):
            yield self._start_date + timedelta(i)


class EvenDateTemplate(DateTemplate):
    def get_days(self):
        for i in range(self._length):
            date_ = self._start_date + timedelta(i)

            if date_.day % 2 == 0:
                yield date_


class OddDateTemplate(DateTemplate):
    def get_days(self):
        for i in range(self._length):
            date_ = self._start_date + timedelta(i)

            if date_.day % 2 == 1:
                yield date_


class WeekdayDateTemplate(DateTemplate):
    def __init__(self, start_date, length, weekday):
        self.weekday = weekday

        DateTemplate.__init__(self, start_date, length)

    def get_text(self, lang):
        return WeekdayDateTemplate.get_weekday_text(self.weekday, lang)

    @classmethod
    def get_weekday_text(cls, weekday, lang):
        return xgettext_weekday_short(weekday, lang)

    def get_days(self):
        for i in range(self._length):
            date_ = self._start_date + timedelta(i)

            if date_.isoweekday() == self.weekday:
                yield date_


class HolidayDateTemplate(DateTemplate):
    def get_days(self):
        return YCalendar.get_weekends(self._start_date, self._start_date + timedelta(self._length + 5))


class WorkdayDateTemplate(DateTemplate):
    def get_days(self):
        return YCalendar.get_workdays(self._start_date, self._start_date + timedelta(self._length + 5))


class UnionDateTemplate(DateTemplate):
    def __init__(self, *templates):
        self._start_date = templates[0]._start_date
        self._length = templates[0]._length

        self.texts = {}
        self.days = set()

        for t in templates:
            if (t._start_date != self._start_date or
                    t._length != self._length):
                raise ValueError('Templates lenghts and start dates must match')

            self.days.update(t.days)

        # Для шаблонов типа "кроме вт, пт"
        if len(templates) in (5, 6) and all(isinstance(t, WeekdayDateTemplate) for t in templates):
            weekdays = set(t.weekday for t in templates)

            except_weekdays = [weekday for weekday in xrange(1, 8) if weekday not in weekdays]

            for lang in settings.FRONTEND_LANGUAGES:
                template = get_datetemplate_translation(lang, 'UnionDateTemplate_except')

                self.texts[lang] = stringify(xformat(template, **{
                    'except': u', '.join(WeekdayDateTemplate.get_weekday_text(weekday, lang)
                                         for weekday in except_weekdays)
                }))

            self.has_except = True

        else:
            for lang in settings.FRONTEND_LANGUAGES:
                self.texts[lang] = u', '.join(t.get_text(lang) for t in templates)

    def get_text(self, lang):
        return self.texts[lang]


def _get_combos(lst, depth):
    if len(lst) == 0 or len(lst) < depth:
        return []

    if len(lst) == depth:
        return [lst]

    if depth <= 0:
        return None

    combos = []

    if depth == 1:
        for item in lst:
            combos.append([item])
        return combos
    else:
        for index, item in enumerate(lst[:-1]):
            for combo in _get_combos(lst[index + 1:], depth - 1):
                combos.append([item] + combo)
        return combos


def _all_combos(templates):
    for comb_depth in range(2, len(templates)):
        for combo in _get_combos(templates, comb_depth):
            yield UnionDateTemplate(*combo)


@cache_until_switch
def _get_basic_templates(start_date, length):
    log.debug('Generating basic templates (%s, %s)' % (start_date, length))

    basic_templates = [
        AllDateTemplate(start_date, length),
        EvenDateTemplate(start_date, length),
        OddDateTemplate(start_date, length),
    ]

    weekday_templates = [WeekdayDateTemplate(start_date, length, d) for d in range(1, 8)]
    weekday_combined = _all_combos(weekday_templates)

    templates = chain(basic_templates,
                      weekday_templates, weekday_combined)

    return list(templates)


@cache_until_switch
def _get_all_templates(start_date, length):
    log.debug('Generating all templates (%s, %s)' % (start_date, length))
    templates = [
        AllDateTemplate(start_date, length),
        EvenDateTemplate(start_date, length),
        OddDateTemplate(start_date, length),
        HolidayDateTemplate(start_date, length),
    ] + [WeekdayDateTemplate(start_date, length, d) for d in range(1, 8)]

    combo_templates = []

    for comb_depth in range(2, len(templates)):
        for combo in _get_combos(templates, comb_depth):
            combo_templates.append(UnionDateTemplate(*combo))

    return templates + combo_templates


@cache_until_switch
def _get_buses_templates(start_date, length):
    log.debug('Generating buses templates (%s, %s)' % (start_date, length))
    basic_templates = [
        AllDateTemplate(start_date, length),
        EvenDateTemplate(start_date, length),
        OddDateTemplate(start_date, length),
    ]

    weekday_templates = [WeekdayDateTemplate(start_date, length, d) for d in range(1, 8)]
    weekday_combined = _all_combos(weekday_templates)

    templates = chain(weekday_templates, weekday_combined, basic_templates)

    return list(templates)


_template_generators = {
    'basic': _get_basic_templates,
    'all': _get_all_templates,
    'buses': _get_buses_templates
}


def commaseparated_datelist(dates):
    return u", ".join([date_.strftime("%d.%m") for date_ in sorted(dates)])


class Matcher(object):
    max_mismatch_days = 3
    min_match_days = 7
    min_schedule_length_in_range = 8
    search_first_day_in_past_days = 0

    def __init__(self, templates, today):
        self.today = today
        self.templates = templates

    def get_template_first_day(self, mask):
        search_from_day = self.today - timedelta(days=self.search_first_day_in_past_days)
        # Первый день хождения после сегодняшнего дня или сегодня(RASP-2549) по умолчанию
        # Можно выбирать на сколько дней назад заглядывать за первой датой хождения
        try:
            return (d for d in mask.iter_dates() if d >= search_from_day).next()
        except StopIteration:
            # Рейс не ходит после сегодняшнего дня
            return None

    @cache_method_result
    def find_template(self, mask, schedule_length,
                      schedule_plan=None, min_match_days=None):

        min_match_days = min_match_days or self.min_match_days

        # в связи с тем, что у нас циклический календарь, может возникнуть ситуация, что последний день хождения нитки
        # очень далеко (например, сейчас январь и у нас расписание еще за прошлый год,
        # т.е. последний день хождения - в декабре)
        # поэтому надо от маски отфильтровать такие дни - взять только первые полгода
        # @see https://jira.yandex-team.ru/browse/RASP-5227

        first = self.get_template_first_day(mask)
        if first is None:
            return None, None, None

        # Берем примерно полгода вперед, чтобы не было перекрытий с конца года
        run_days = [d for d in mask.dates() if d <= first + timedelta(days=150)]

        # Последняя дата, учитываемая при наложении шаблонов
        last = run_days[-1] + timedelta(1)

        range_days = self.get_cached_range_days(first, last)

        run_days_set = set(run_days) & range_days
        run_days_in_range = sorted(run_days_set)
        schedule_length_in_range = (run_days_in_range[-1] - run_days_in_range[0]).days + 1  # + 1 т.к. включительно

        if len(run_days_in_range) < min_match_days or schedule_length_in_range < self.min_schedule_length_in_range:
            return None, None, None

        template, mismatch, m_type = self.best_match(run_days_set, range_days)

        if not template:
            return None, None, None

        if len(mismatch) > self.max_mismatch_days:
            return None, None, None

        extrapolable = True

        params = {}

        # Сравниваем вместо первого дня хождения нитки первый день графика
        if schedule_plan and schedule_plan.start_date > self.today:
            pre_days = set(daterange(self.today, schedule_plan.start_date))

            # У маски шаблона есть дни хождения в диапазоне [сегодня, первый день хождения нитки)
            if template.days & pre_days:
                params['start'] = '%02d.%02d' % (schedule_plan.start_date.day, schedule_plan.start_date.month)

        elif first > self.today:
            pre_days = set(daterange(self.today, first))

            # У маски шаблона есть дни хождения в диапазоне [сегодня, первый день хождения нитки)
            if template.days & pre_days:
                params['start'] = '%02d.%02d' % (first.day, first.month)

        # Сравниваем вместо последнего дня хождения нитки последний день графика
        if schedule_plan and schedule_plan.end_date < self.today + timedelta(schedule_length):
            post_days = set(daterange(schedule_plan.end_date + timedelta(1),
                                      self.today + timedelta(schedule_length)))

            if template.days & post_days:
                extrapolable = False

                # У маски шаблона есть дни хождения в диапазоне
                # (последний день хождения нитки, последний день достоверного расписания)
                params['end'] = '%02d.%02d' % (schedule_plan.end_date.day, schedule_plan.end_date.month)

        elif run_days[-1] < self.today + timedelta(schedule_length):
            post_days = set(daterange(run_days[-1] + timedelta(1), self.today + timedelta(schedule_length)))

            if template.days & post_days:
                extrapolable = False

                last = run_days[-1]

                # У маски шаблона есть дни хождения в диапазоне
                # (последний день хождения нитки, последний день достоверного расписания)
                params['end'] = '%02d.%02d' % (last.day, last.month)

        if mismatch:
            extrapolable = False

            if m_type == 'except':
                params['except'] = commaseparated_datelist(mismatch)

            else:
                params['extra'] = commaseparated_datelist(mismatch)

        texts = {}

        if params:
            key_tokens = []

            for key in ['start', 'end', 'except', 'extra']:
                if key in params:
                    key_tokens.append(key)

            template_key = "Matcher_%s" % ("_".join(key_tokens))

            for lang in settings.FRONTEND_LANGUAGES:
                params['main'] = template.get_text(lang)

                texts[lang] = stringify(xformat(get_datetemplate_translation(lang, template_key), **params))

        else:
            for lang in settings.FRONTEND_LANGUAGES:
                texts[lang] = template.get_text(lang)

        return texts, template, extrapolable

    def best_match(self, run_days, range_days):
        best = None
        best_mismatch = None
        best_m_type = None

        for t in self.templates:
            # Дни маски, в которые рейсы не ходит
            except_ = (t.days - run_days).intersection(range_days)

            # Дни хождения, дополнительные к маске
            extra = (run_days - t.days).intersection(range_days)

            # Точное совпадение
            if not except_ and not extra:
                return t, set(), None

            # Есть и те и другие, нельзя сформировать строку
            if except_ and extra:
                continue

            # "кроме" уже есть в маске
            if t.has_except:
                continue

            if except_:
                mismatch = except_
                m_type = 'except'
            elif extra:
                mismatch = extra
                m_type = 'extra'

            if not best_mismatch or len(mismatch) < len(best_mismatch):
                best = t
                best_mismatch = mismatch
                best_m_type = m_type

        return best, best_mismatch, best_m_type

    @cache_method_result
    def get_cached_range_days(self, first, last):
        return set(daterange(first, last))


def get_generator(generator_name):
    try:
        return _template_generators[generator_name]
    except KeyError:
        raise ValueError('Template generator {} not found'.format(generator_name))


def get_matcher(generator_name, today, start_date, length):
    generator = get_generator(generator_name)

    return Matcher(generator(start_date, length), today)
