# -*- coding: utf-8 -*-

"""Модуль для разбора форматов дат, предусмотренных проектом"""

import json
import re
from datetime import date, timedelta

from django.conf import settings

from common.utils.text import normalize, punto_variants, NORM_CHAR_MAP
from common.xgettext.common import (
    xgettext_month, xgettext_month_short, xgettext_month_genitive, get_datetemplate_translation, xgettext_weekday
)
from common.xgettext.i18n import gettext


MONTH_NAMES = {
    u'январь': 1,
    u'февраль': 2,
    u'март': 3,
    u'апрель': 4,
    u'май': 5,
    u'июнь': 6,
    u'июль': 7,
    u'август': 8,
    u'сентябрь': 9,
    u'октябрь': 10,
    u'ноябрь': 11,
    u'декабрь': 12,
}

# Короткие названия
for name, value in MONTH_NAMES.items():
    MONTH_NAMES[name[:3]] = value

MONTH_NAMES.update({
    u'января': 1,
    u'февраля': 2,
    u'марта': 3,
    u'апреля': 4,
    u'мая': 5,
    u'июня': 6,
    u'июля': 7,
    u'августа': 8,
    u'сентября': 9,
    u'октября': 10,
    u'ноября': 11,
    u'декабря': 12,
})


class DataParserInfo(object):
    # Дата
    # RASP-5209
    # 20 (ближайшая дата с таким днем)

    DAY = r'(\d{1,2})'

    # RASP-1216
    # 20.9.2008 (20.09.2008, 20.9.08)
    # 20/9/2008 (20/09/2008, 20/9/08)
    # 20-9-2008 (20-09-2008, 20-9-08)
    # RASP-1307
    # 09.12 (год - текущий)

    DIGITAL = r'(\d{1,2})[./-](\d{1,2})(?:[./-](\d{4}|\d\d))?'

    # 2008-09-01 00:00:00

    ANSI = r'(\d{4})-(\d{1,2})-(\d{1,2})(?:\s(\d{2}):(\d{2}):(\d{2}))?'

    def __init__(self, month_names, human_pointers, human_weekdays):
        self.month_names = month_names

        # 20 сентября
        # 20 сентября 2008
        # 20 сентября 08

        self.months = "(?:" + "|".join(self.month_names.keys()) + ")"
        self.human = r'(\d{1,2})\s*(%s)(?:\s+(\d{4}|\d\d))?' % self.months

        # RASP-695:
        # вчера
        # сегодня
        # завтра
        # послезавтра
        # через неделю
        # через месяц

        self.human_pointer = '(%s)' % '|'.join(r for d, r in human_pointers)
        self.human_pointers = [(d, re.compile(expression + '$', re.U)) for d, expression in human_pointers]

        # RASP-3846
        # понедельник
        # вторник
        # среда
        # четверг
        # пятница
        # суббота
        # воскресенье

        self.human_weekday = '(%s)' % '|'.join(r for d, r in human_weekdays)
        self.human_weekdays = [(d - 1, re.compile(r, re.U)) for d, r in human_weekdays]

        self.digital_or_human_or_pointer = '(%s)' % '|'.join([DataParserInfo.DIGITAL, self.human, self.human_pointer])

        # Функция, компилирующая регекспы
        def handler(expression, handler):  # @NoSelf
            regex = re.compile(expression + '$', re.U)
            return regex, handler, expression

        self.handlers = [
            handler(self.DAY, 'day_handler'),
            handler(self.DIGITAL, 'digital_handler'),
            handler(self.ANSI, 'ansi_handler'),
            handler(self.human, 'human_handler'),
            handler(self.human_pointer, 'human_pointer_handler'),
            handler(self.human_weekday, 'human_weekday_handler'),
        ]


def _get_month_names():
    result = {}

    for lang in settings.FRONTEND_LANGUAGES:
        for month_number in range(1, 13):
            for translation in (
                xgettext_month,
                xgettext_month_short,
                xgettext_month_genitive
            ):
                value = normalize(translation(month_number, lang))

                result[value] = month_number

    return result


def _get_weekdays_names():
    result = []

    for lang in settings.FRONTEND_LANGUAGES:
        for weekday_number in range(1, 8):
            weekday = xgettext_weekday(weekday_number, lang)

            result.append((weekday_number, normalize(weekday)))

    return set(result)


def _get_human_pointer(lang, key):
    return get_datetemplate_translation(lang, key).replace(' ', '\s+').strip(' \t\n\r')


def _get_human_pointers():
    values = []

    for lang in settings.FRONTEND_LANGUAGES:
        values.extend([
            (-2, _get_human_pointer(lang, "shift-2d")),
            (-1, _get_human_pointer(lang, "shift-1d")),
            (0, gettext(u"сегодня", lang=lang)),
            (1, _get_human_pointer(lang, "shift+1d")),
            (2, _get_human_pointer(lang, "shift+2d")),
            ('week', _get_human_pointer(lang, "shift+week")),
            ('month', _get_human_pointer(lang, "shift+month")),
            (None, _get_human_pointer(lang, "shift+none")),
        ])

    # Дополнительные значения (например, послезавтра можно сказать "yarın gün sonra" и "yarından sonraki gün", но через
    # _get_human_pointer получаем только одно значение "yarın gün sonra")
    values.extend([
        (1, u'yarın'),
        (2, u'yarından\s+sonraki\s+gün'),
    ])

    result = []

    for shift, value in values:
        result.append((shift, normalize(value)))

    return set(result)


DATEPARSER_INFO = DataParserInfo(
    month_names=_get_month_names(),
    human_pointers=_get_human_pointers(),
    human_weekdays=_get_weekdays_names()
)


class DateParser(object):
    def __init__(self, data, today=None, past_border=None, base_date=None, parser_info=DATEPARSER_INFO):
        self.parser_info = parser_info
        if today is None:
            today = date.today()
        self.today = today
        self.past_border = past_border
        self.base_date = base_date or today  # Дата, от которой отсчитываются интервалы "через неделю" и "через месяц"
        self.dates = self.parse(data)

    # Собственно функция-парсер
    def parse(self, data):
        try:
            data = normalize(data.strip())
        except AttributeError:
            raise TypeError('Cannot parse None')

        try:
            return self._parse(data)
        except ValueError:
            pass

        for variant in punto_variants(data):
            try:
                return self._parse(variant)
            except ValueError:
                pass

        raise ValueError(u'Unknown date format %r' % data)

    def _parse(self, data):
        for regex, handler, expression in self.parser_info.handlers:
            m = regex.match(data)
            if m:
                return getattr(self, handler)(*m.groups())

        raise ValueError(u'Unknown date format %r' % data)

    def tuple_to_date(self, day, month=None, year=None):
        """
        t - список (day, month, year) или (day, month), где
        day, month, year - числовые значения года или месяца, возможно - строки
        возвращает datet
        """
        day = int(day)
        month = month and int(month)
        current_year = self.today.year

        if not month:
            if day < self.today.day:
                month = self.today.month + 1
            else:
                month = self.today.month
            if month > 12:
                current_year += 1
                month = 1

        # Если нужно, добавляем год
        if year is None:
            year = current_year
        else:
            year = int(year)

            if year < 100:
                year = current_year - (current_year % 100) + year

        try:
            d = date(year, month, day)
        except ValueError:
            if (month, day) == (2, 29):
                d = date(year + 1, month, day)
            else:
                raise

        # Если дата более чем на past_border времени в прошлом, то она пересчитывается в следующем году
        if self.today and self.past_border:
            if self.today - d > self.past_border:
                return date(year + 1, int(month), int(day))

        return d

    def day_handler(self, *data):
        """Обработка даты с днем"""
        return self.tuple_to_date(*data)

    def digital_handler(self, *data):
        """Обработка даты с цифрами"""
        return self.tuple_to_date(*data)

    def ansi_handler(self, *data):
        """Обработка даты с цифрами в формате ansi"""
        return self.tuple_to_date(data[2], data[1], data[0])

    def human_handler(self, day, month, year):
        """Обработка даты с алфавитным месяцем, на входе: день, месяц[, год]"""

        # обрабатываем месяц
        month = self.parse_month(month)

        return self.tuple_to_date(day, month, year)

    def human_pointer_handler(self, text):
        delta = None

        for d, r in self.parser_info.human_pointers:
            if r.match(text):
                delta = d
                break

        if delta is None:
            return None

        if delta == 'week':
            return self.base_date + timedelta(days=7)

        if delta == 'month':
            day = self.base_date.day
            month = self.base_date.month + 1
            year = self.base_date.year

            if month > 12:
                year += 1
                month = 1

            try:
                return date(year, month, day)
            except ValueError:
                # день не умещается в месяц
                return date(year, month, 28)  # а черт с ними, с високосными годами

        return self.today + timedelta(days=delta)

    def human_weekday_handler(self, text):
        for d, r in self.parser_info.human_weekdays:
            if r.match(text):
                delta = d - self.today.weekday()  # Разница между текущим днем недели и нужным

                if delta < 0:  # Нужна следующая неделя
                    delta += 7

                return self.today + timedelta(days=delta)

        raise ValueError(u'Unknown day %r' % text)

    def parse_month(self, name):
        try:
            return self.parser_info.month_names[name.lower()]
        except KeyError:
            raise ValueError(u'Unknown month %r' % name)


def parse_human_date(*args, **kwargs):
    return DateParser(*args, **kwargs).dates


# Автоматический генератор валидирующего js
def generate_validator_js():
    parsers = ["""\
    {
        expression: /^%s$/i,
        handler: '%s'
    }""" % (expression, handler)
        for regex, handler, expression in DATEPARSER_INFO.handlers
    ]

    char_map = json.dumps(NORM_CHAR_MAP, ensure_ascii=False)

    return u"""\
// Autogenerated by lib/dateparser.py, do not edit!

var charMap =  %s;

var stripRegexp = /^\s*(.*?)\s*$/,
    parsers,
    handlers;

parsers = [
    %s
];

handlers = {
    'human_handler': function (day, month, year) {
        // Месяц правильный подразумевается из регекспа
        if(day < 1 || day > 31) {
            return false;
        }

        return true;
    },
    'digital_handler': function (day, month, year) {
        if(day < 1 || day > 31 || month < 1 || month > 12) {
            return false;
        }

        return true;
    }
}

function checkDate(value) {
    var i,
        result,
        variants;

    // Убираем пробелы в начале и конце
    value = stripRegexp.exec(value)[1];

    // IE<9 не воспринимает неразрывный пробел как \s
    value = value.replace(/\\u00a0/g, ' ');

    $.each(charMap, function(k, v) {
        value = value.split(k).join(v);
    });

    variants = [value];

    // Варинты с коррекцией раскладки
    Array.prototype.push.apply(variants, puntoVariants(value));

    for(i = 0; i < variants.length; i++) {
        result = _checkDate(variants[i]);

        if (result) {
            return true;
        }
    }

    return false;
}

function _checkDate(value) {
    // Проходим по всем регулярным выражениям, если какое-то совпадает,
    // значит формат даты - известен
    for(var i = 0; i < parsers.length; i++) {
        var parser = parsers[i],
            match = parser.expression.exec(value);

        if(match) {
            if(parser.handler in handlers) {
                return handlers[parser.handler].apply(null, match.slice(1));
            } else {
                return true;
            }
        };
    }

    return false;
};

var enKeys = "qwertyuiop[{]}asdfghjkl;:\\'\\"zxcvbnm,<.>`"; // enKeys длиннее из-за экранирования (\\)
var ruKeys = "йцукенгшщзххъъфывапролджжээячсмитьббююё";
var ukKeys = "йцукенгшщзххїїфівапролджжєєячсмитьббююё";

function puntoVariants(value) {
    var i, j, k,
        baseKeysSet = [enKeys],
        baseKeys,
        localKeysSet = [ruKeys, ukKeys],
        localKeys,
        variants = [];

    function convert(fromKeys, toKeys) {
        var i,
            converted = value;

        for(i = 0; i < fromKeys.length; i++) {
            converted = converted.split(fromKeys.charAt(i)).join(toKeys.charAt(i));
        }

        variants.push(converted);
    }

    for(i = 0; i < baseKeysSet.length; i++) {
        baseKeys = baseKeysSet[i];

        for(j = 0; j < localKeysSet.length; j++) {
            localKeys = localKeysSet[j];

            convert(baseKeys, localKeys);
            convert(localKeys, baseKeys);
        }
    }

    return variants;
}
""" % (char_map, ',\n    '.join(parsers))
