# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import gzip
import logging
import tempfile
from contextlib import closing
from subprocess import Popen, PIPE

import requests

from common.utils.progress import PercentageStatus

log = logging.getLogger(__name__)


LISTING_TIMEOUT = 10
DOWNLOAD_TIMEOUT = 5 * 60  # 5 минут
CHUNK_SIZE = 1024 * 1014

TIMESTAMP_TABLE_NAME = 'load_dump_timestamp'


def download_dump(dump_url):
    dump_file = tempfile.NamedTemporaryFile(suffix='-rasp-dump.sql.gz')

    log.info(u'Загружаем %s в %s...', dump_url, dump_file.name)

    resp = requests.head(dump_url, timeout=LISTING_TIMEOUT, verify=False)
    filesize = int(resp.headers['content-length'])
    log.info(u'Размер %s %s', dump_url, filesize)
    download_status = PercentageStatus(filesize, log)

    with closing(
        requests.get(dump_url, timeout=DOWNLOAD_TIMEOUT, stream=True, verify=False)
    ) as resp:
        for chunk in resp.iter_content(CHUNK_SIZE):
            dump_file.write(chunk)
            download_status.step(len(chunk))

    log.info(u'Загрузили %s в %s', dump_url, dump_file.name)

    return dump_file


class DbDumpManager(object):
    def __init__(self, db_name, exclude_tables=(), connection=None):
        self.db_name = db_name
        self.exclude_tables = exclude_tables

        if connection is None:
            from django.db import connection as default_conn
            self.connection = default_conn
        else:
            self.connection = connection
        self.connection.ensure_connection()

    def update_database_from_dump_url(self, dump_url):
        host = self.connection.connection.conn_params['host']
        log.info(u'Загружаем дамп %s в базу %s -> %s...', dump_url, host, self.db_name)

        dump_file = download_dump(dump_url)
        self.purge_database()
        self.load_dump_to_database(dump_file)
        self.save_dump_url(dump_url)

    def database_exists(self):
        cursor = self.connection.cursor()
        cursor.execute('SHOW DATABASES')
        db_names = [r[0] for r in cursor.fetchall()]
        return self.db_name in db_names

    def purge_database(self):
        log.info(u'Пересоздаем %s...', self.db_name)

        qn = self.connection.ops.quote_name
        cursor = self.connection.cursor()

        if self.database_exists():
            cursor.execute("DROP DATABASE {}".format(qn(self.db_name)))

        cursor.execute("CREATE DATABASE {} DEFAULT CHARSET utf8".format(qn(self.db_name)))

    def load_dump_to_database(self, dump_file):
        log.info('Загружаем файл %s в базу %s...', dump_file.name, self.db_name)

        log.info('Считаем количество строк в файле...')
        line_count = 0
        dump_file.seek(0)
        with gzip.GzipFile(fileobj=dump_file, mode='rb') as f:
            for l in f:
                line_count += 1

        log.info('Всего %s строк', line_count)

        conn_params = self.connection.connection.conn_params
        process = Popen([
            'mysql',
            '--max-allowed-packet=128M',
            '--net-buffer-length=8M',
            '--user={}'.format(conn_params['user']),
            '--host={}'.format(conn_params['host']),
            '--password={}'.format(conn_params['passwd']),
            self.db_name
        ], stdin=PIPE)

        exclude_lines = tuple(
            'INSERT INTO `{}`'.format(t) for t in self.exclude_tables
        )

        if not exclude_lines:
            log.info('Загружаем все без исключений')
        else:
            for line in exclude_lines:
                log.info('Исключаем записи начинающиется на "%s"', line)

        exclude_lines = tuple(el.encode('utf-8') for el in exclude_lines)

        status = PercentageStatus(line_count, log)

        dump_file.seek(0)
        with gzip.GzipFile(fileobj=dump_file, mode='rb') as f:
            for l in f:
                if l.startswith(b'CREATE TABLE'):
                    log.info(l[:-2])

                if not l.startswith(exclude_lines):
                    process.stdin.write(l)

                status.step(1)

        process.stdin.close()
        retcode = process.wait()

        if retcode != 0:
            raise Exception('Dump was not loaded')

        self.connection.close()

    def save_dump_url(self, dump_url):
        log.info('Записываем dump_url в %s...', self.db_name)

        qn = self.connection.ops.quote_name
        cursor = self.connection.cursor()

        table_name = '{}.{}'.format(qn(self.db_name), qn(TIMESTAMP_TABLE_NAME))

        cursor.execute('''
            CREATE TABLE {} (`dump_url` varchar (255) PRIMARY KEY)
            DEFAULT CHARSET utf8
        '''.format(table_name))

        cursor.execute('''
            INSERT INTO {} (`dump_url`) VALUES (%s)
        '''.format(table_name), [dump_url])

        self.connection.connection.commit()

    def list_tables(self):
        qn = self.connection.ops.quote_name
        cursor = self.connection.cursor()
        cursor.execute('SHOW TABLES FROM {}'.format(qn(self.db_name)))
        return [r[0] for r in cursor.fetchall()]

    def load_dump_url(self):
        if not (database_exists(self.db_name) and self.timestamp_table_exists()):
            return

        qn = self.connection.ops.quote_name

        cursor = self.connection.cursor()
        cursor.execute("SELECT `dump_url` FROM {}.{}".format(
            qn(self.db_name), qn(TIMESTAMP_TABLE_NAME)
        ))

        rows = cursor.fetchall()
        if not rows:
            return

        return rows[0][0]

    def timestamp_table_exists(self):
        return TIMESTAMP_TABLE_NAME in list_tables(self.db_name)


# ====== оставленные для обратной совместимости функции с дефолтным connection =======
def update_database_from_dump_url(db_name, dump_url, exclude_tables=()):
    return DbDumpManager(db_name, exclude_tables=exclude_tables).update_database_from_dump_url(dump_url)


def purge_database(db_name):
    return DbDumpManager(db_name).purge_database()


def database_exists(db_name):
    return DbDumpManager(db_name).database_exists()


def load_dump_to_database(dump_file, db_name, exclude_tables=()):
    return DbDumpManager(db_name, exclude_tables=exclude_tables).load_dump_to_database(dump_file)


def list_tables(db_name):
    return DbDumpManager(db_name).list_tables()


def load_dump_url(db_name):
    return DbDumpManager(db_name).load_dump_url()

# ====== /оставленные для обратной совместимости функции с дефолтным connection =======
