# coding: utf8

import hashlib
import inspect
import logging
import pickle
import zlib
from datetime import datetime
from email.utils import parsedate
from functools import wraps

from django.core.cache import cache
from django.http import HttpResponseNotModified

from common.models.timestamp import Timestamp
from travel.rasp.library.python.common23.date import environment
from common.utils.caching import get_package_cache_root
from common.utils.date import MSK_TZ, UTC_TZ


log = logging.getLogger(__name__)


def _parse_if_modified_since(request):
    try:
        if_modified_since = UTC_TZ.localize(parse_http_datetime(request.META['HTTP_IF_MODIFIED_SINCE']))
    except KeyError:
        if_modified_since = None
    except Exception:
        log.exception("Can't parse IF_MODIFIED_SINCE")
        if_modified_since = None

    return if_modified_since


def modified_at(last_modified_func):
    """
    :param last_modified_func: функция, возвращающая локализованную дату изменения
    """

    def decorator(func):
        @wraps(func)
        def wrapper(request, *args, **kwargs):
            last_modified = last_modified_func(request, *args, **kwargs)

            if_modified_since = _parse_if_modified_since(request)
            if if_modified_since is not None and if_modified_since >= last_modified:
                return HttpResponseNotModified()

            response = func(request, *args, **kwargs)

            # TODO: возможно нужно будет заточить Cache-Control тоже
            response['Last-Modified'] = format_http_datetime(last_modified)

            return response

        return wrapper

    return decorator


def call_hash(func, request):
    assert request.method == 'GET'

    params = sorted(request.GET.items())
    s = '{}:{}:{}:{}'.format(inspect.getfile(func), func.__name__, request.path, str(params))
    return hashlib.md5(s).hexdigest()


def modified_by_timestamp_cached(timestamp_code, timeout=None, hash_func=None):
    """
    Поддержка функциональности Last-Modified и If-Modified-Since с кэшированием.
    Зависит от Timestamp в базе.

    Декорируемая функция должна возвращать django.http.HttpResponse

    :param timestamp_code: код объекта из common.models.timestamp.Timestamp
    """
    if not hash_func:
        hash_func = call_hash

    def decorator(func):

        @wraps(func)
        def wrapper(request, *args, **kwargs):
            def get_key():
                return '{}httpcaching.py/{}'.format(get_package_cache_root(), hash_func(func, request))

            def get_cache():
                key = get_key()
                try:
                    zipped = cache.get(key)
                except Exception as ex:
                    log.exception('Ошибка получения кеша. Ключ {}. Ошибка: {}'.format(key, repr(ex)))
                    return {'last_checked': None, 'response': None}

                if zipped:
                    return pickle.loads(zlib.decompress(zipped))
                else:
                    return {'last_checked': None, 'response': None}

            def set_cache(data):
                zipped_data = zlib.compress(pickle.dumps(data))
                cache.set(get_key(), zipped_data, timeout)

            # Если пришел if_modified_since, то мы можем отдавать 304,
            # иначе эта логика пропускается.
            if_modified_since = _parse_if_modified_since(request)
            db_timestamp = get_timestamp(timestamp_code)
            # данные не изменились с запрашиваемой даты
            if if_modified_since and if_modified_since >= db_timestamp:
                return HttpResponseNotModified()

            # проверяем, нужно ли обновлять кэш, и изменились ли данные
            cached = get_cache()
            if cached['last_checked'] is None or cached['last_checked'] < db_timestamp:
                new_response = func(request, *args, **kwargs)
                if not cached['response'] or cached['response'].content != new_response.content:
                    cached['response'] = new_response
                    cached['last_modified'] = db_timestamp

                cached['last_checked'] = environment.now_aware()
                set_cache(cached)

            if if_modified_since and if_modified_since >= cached['last_modified']:
                return HttpResponseNotModified()
            else:
                response = cached['response']
                response['Last-Modified'] = format_http_datetime(cached['last_modified'])

                return response

        return wrapper

    return decorator


def format_http_datetime(dt):
    """ Принимает время с таймзоной и возвращает в виде строки по RFC2616.
    http://tools.ietf.org/html/rfc2616#section-14.29
    """
    dt = dt.astimezone(UTC_TZ)
    return dt.strftime('%a, %d %b %Y %H:%M:%S GMT')


def parse_http_datetime(s):
    return datetime(*parsedate(s)[:6])


def get_timestamp(timestamp_name):
    """
    В Timestamp времена сохраняются без таймзоны в московском времени.
    """
    try:
        dt = Timestamp.get(timestamp_name)
    except Timestamp.DoesNotExist:
        dt = None

    return MSK_TZ.localize(dt) if dt else environment.now_aware()


def prepare_all_timestamp(*args, **kwargs):
    return get_timestamp('prepare_all')


modified_by_prepare_all = modified_at(prepare_all_timestamp)
